# Number Formatting Guide - Indonesian Format

## Overview

The application now includes global number formatting helpers for **Indonesian format**:
- **Thousand separator**: Dot (`.`)
- **Decimal separator**: Comma (`,`)

Example: `1.000.000,50` = 1 million and 50 cents

---

## PHP Helpers (Backend)

Located in: [`app/Helpers/Helpers.php`](app/Helpers/Helpers.php)

### 1. `formatNumber($number, $decimals = 0)`

Format a number with Indonesian thousand separator.

```php
use App\Helpers\Helpers;

Helpers::formatNumber(1000);           // "1.000"
Helpers::formatNumber(1000000);        // "1.000.000"
Helpers::formatNumber(1234.56, 2);     // "1.234,56"
Helpers::formatNumber(1234567.89, 2);  // "1.234.567,89"
```

### 2. `parseNumber($formattedNumber)`

Parse Indonesian formatted number back to float.

```php
use App\Helpers\Helpers;

Helpers::parseNumber("1.000");         // 1000
Helpers::parseNumber("1.000.000");     // 1000000
Helpers::parseNumber("1.234,56");      // 1234.56
```

### 3. `formatCurrency($number, $decimals = 0)`

Format number as Indonesian Rupiah currency.

```php
use App\Helpers\Helpers;

Helpers::formatCurrency(1000);         // "Rp 1.000"
Helpers::formatCurrency(1000000);      // "Rp 1.000.000"
Helpers::formatCurrency(1234.56, 2);   // "Rp 1.234,56"
```

---

## JavaScript Helpers (Frontend)

Located in: [`resources/assets/js/number-formatter.js`](resources/assets/js/number-formatter.js)

### 1. `formatNumber(number, decimals = 0)`

Format a number with Indonesian thousand separator.

```javascript
formatNumber(1000);           // "1.000"
formatNumber(1000000);        // "1.000.000"
formatNumber(1234.56, 2);     // "1.234,56"
formatNumber(1234567.89, 2);  // "1.234.567,89"
```

### 2. `parseNumber(formattedNumber)`

Parse Indonesian formatted number back to float.

```javascript
parseNumber("1.000");         // 1000
parseNumber("1.000.000");     // 1000000
parseNumber("1.234,56");      // 1234.56
```

### 3. `formatCurrency(number, decimals = 0)`

Format number as Indonesian Rupiah currency.

```javascript
formatCurrency(1000);         // "Rp 1.000"
formatCurrency(1000000);      // "Rp 1.000.000"
formatCurrency(1234.56, 2);   // "Rp 1.234,56"
```

---

## Automatic Number Formatting in Forms

### Method 1: Using `data-format="number"` (Recommended)

Add `data-format="number"` attribute to any text or number input to automatically format as user types.

```blade
<!-- Basic number input with automatic formatting -->
<input type="text"
       class="form-control"
       name="distance"
       data-format="number"
       value="{{ old('distance', $model->distance) }}">

<!-- Number input with 2 decimal places -->
<input type="text"
       class="form-control"
       name="price"
       data-format="number"
       data-decimals="2"
       value="{{ old('price', $model->price) }}">
```

**What it does:**
- ✅ Automatically formats as user types (1000 → 1.000)
- ✅ Shows formatted value with thousand separators
- ✅ Handles decimal places if specified
- ✅ Maintains cursor position while typing

**When to use:**
- Distance, duration, weight, quantity fields
- Any numeric field that needs thousand separators
- Price, amount, salary fields with decimals

---

### Method 2: Using `data-format="currency"` for Money

For currency/money fields with "Rp" prefix:

```blade
<input type="text"
       class="form-control"
       name="credit_limit"
       data-format="currency"
       data-decimals="2"
       value="{{ old('credit_limit', $model->credit_limit) }}">
```

**Result:**
- User types: `1000000`
- Shows: `Rp 1.000.000`

---

### Method 3: Submit Raw Values

If you want to submit raw numeric values (without thousand separators):

```blade
<input type="text"
       class="form-control"
       name="price"
       data-format="number"
       data-decimals="2"
       data-submit-raw="true"
       value="{{ old('price', $model->price) }}">
```

**What `data-submit-raw="true"` does:**
- User sees: `1.234,56`
- Form submits: `1234.56` (raw value)
- Backend receives clean numeric value

---

## Examples for Different Field Types

### 1. Distance Field (No Decimals)

```blade
<div class="col-md-6">
    <label for="distance" class="form-label">
        Distance (KM) <span class="text-danger">*</span>
    </label>
    <input type="text"
           class="form-control @error('distance') is-invalid @enderror"
           id="distance"
           name="distance"
           data-format="number"
           data-submit-raw="true"
           value="{{ old('distance', $model->distance) }}"
           required
           placeholder="0">
    @error('distance')
        <div class="invalid-feedback">{{ $message }}</div>
    @enderror
</div>
```

**User Experience:**
- User types: `150`
- Shows: `150`
- User types: `1500`
- Shows: `1.500`
- Submits: `1500` (raw value)

---

### 2. Price Field (2 Decimals)

```blade
<div class="col-md-6">
    <label for="price" class="form-label">
        Price <span class="text-danger">*</span>
    </label>
    <div class="input-group">
        <span class="input-group-text">Rp</span>
        <input type="text"
               class="form-control @error('price') is-invalid @enderror"
               id="price"
               name="price"
               data-format="number"
               data-decimals="2"
               data-submit-raw="true"
               value="{{ old('price', $model->price) }}"
               required
               placeholder="0,00">
        @error('price')
            <div class="invalid-feedback">{{ $message }}</div>
        @enderror
    </div>
</div>
```

**User Experience:**
- User types: `1500000`
- Shows: `1.500.000,00`
- Submits: `1500000` (or `1500000.00`)

---

### 3. Credit Limit with Currency Format

```blade
<div class="col-md-6">
    <label for="credit_limit" class="form-label">Credit Limit</label>
    <input type="text"
           class="form-control @error('credit_limit') is-invalid @enderror"
           id="credit_limit"
           name="credit_limit"
           data-format="currency"
           data-submit-raw="true"
           value="{{ old('credit_limit', $model->credit_limit) }}"
           placeholder="Rp 0">
    @error('credit_limit')
        <div class="invalid-feedback">{{ $message }}</div>
    @enderror
</div>
```

**User Experience:**
- User types: `5000000`
- Shows: `Rp 5.000.000`
- Submits: `5000000`

---

### 4. Toll Fee (Route Example)

```blade
<div class="col-md-4">
    <label for="toll" class="form-label">
        Toll Fee <span class="text-danger">*</span>
    </label>
    <div class="input-group">
        <span class="input-group-text">Rp</span>
        <input type="text"
               class="form-control @error('toll') is-invalid @enderror"
               id="toll"
               name="toll"
               data-format="number"
               data-submit-raw="true"
               value="{{ old('toll', $model->toll) }}"
               required
               placeholder="0">
        @error('toll')
            <div class="invalid-feedback">{{ $message }}</div>
        @enderror
    </div>
</div>
```

---

## Display Formatted Numbers in Views

### In Show/Detail Views

```blade
<tr>
    <th>Distance</th>
    <td>{{ Helpers::formatNumber($model->distance) }} KM</td>
</tr>

<tr>
    <th>Price</th>
    <td>{{ Helpers::formatCurrency($model->price, 2) }}</td>
</tr>

<tr>
    <th>Credit Limit</th>
    <td>{{ Helpers::formatCurrency($model->credit_limit) }}</td>
</tr>
```

### In DataTables

```php
// In controller's getData() method - process after query
$result['data'] = collect($result['data'])->map(function ($item) {
    $item->distance_formatted = Helpers::formatNumber($item->distance);
    $item->price_formatted = Helpers::formatCurrency($item->price, 2);
    return $item;
});
```

Then in DataTable column definition:
```php
["data" => "price_formatted", "title" => "Price", "orderable" => false]
```

---

## Important Notes

### ⚠️ Always Use `data-submit-raw="true"` for Database Fields

When the field will be saved to database, **ALWAYS** add `data-submit-raw="true"`:

```blade
<!-- ✅ CORRECT - submits raw value to database -->
<input type="text"
       name="price"
       data-format="number"
       data-submit-raw="true"
       value="{{ old('price', $model->price) }}">

<!-- ❌ WRONG - submits formatted value "1.000" to database -->
<input type="text"
       name="price"
       data-format="number"
       value="{{ old('price', $model->price) }}">
```

### ✅ Benefits

1. **User-Friendly**: Numbers are easy to read with thousand separators
2. **Automatic**: Formats as user types, no manual intervention
3. **Validation-Friendly**: Submits clean numeric values for backend validation
4. **Cursor-Smart**: Maintains cursor position while formatting
5. **Consistent**: Same format across entire application

### 🎯 Fields That Should Use Formatting

Apply `data-format="number"` to:
- ✅ Distance, duration, quantity
- ✅ Price, amount, salary, credit limit
- ✅ Toll fees, pocket money, bonus
- ✅ Weight, volume, measurements
- ✅ Bank account numbers (maybe without decimals)
- ✅ NPWP, phone numbers (depends on requirements)

### ❌ Fields That Should NOT Use Formatting

Do NOT apply to:
- ❌ ID numbers (KTP, SIM, KK)
- ❌ License plates
- ❌ Codes (truck code, driver code)
- ❌ Dates, times
- ❌ Text fields

---

## Quick Reference

| Use Case | Attribute | Example Input | Display | Submit |
|----------|-----------|---------------|---------|--------|
| Distance | `data-format="number"` `data-submit-raw="true"` | 1500 | 1.500 | 1500 |
| Price | `data-format="number"` `data-decimals="2"` `data-submit-raw="true"` | 1500000 | 1.500.000,00 | 1500000 |
| Currency with Rp | `data-format="currency"` `data-submit-raw="true"` | 5000000 | Rp 5.000.000 | 5000000 |
| Decimal Number | `data-format="number"` `data-decimals="2"` `data-submit-raw="true"` | 123.45 | 123,45 | 123.45 |

---

## Testing

### Test in Browser Console

```javascript
// Format numbers
formatNumber(1000);                 // "1.000"
formatNumber(1000000);              // "1.000.000"
formatNumber(1234.56, 2);           // "1.234,56"

// Parse numbers
parseNumber("1.000");               // 1000
parseNumber("1.234,56");            // 1234.56

// Format currency
formatCurrency(1000000);            // "Rp 1.000.000"
```

### Test in Form

1. Open any form with number fields
2. Type: `1000`
3. Should automatically show: `1.000`
4. Type more: `1000000`
5. Should show: `1.000.000`
6. Submit form
7. Check Network tab - value should be `1000000` (raw)

---

## Summary

✅ **RouteRequest Fixed**
- Toll fee now required and numeric
- Origin and destination can be the same

✅ **PHP Helpers Available**
- `Helpers::formatNumber()`
- `Helpers::parseNumber()`
- `Helpers::formatCurrency()`

✅ **JavaScript Helpers Available**
- `window.formatNumber()`
- `window.parseNumber()`
- `window.formatCurrency()`

✅ **Automatic Formatting**
- Add `data-format="number"` to any input
- Add `data-format="currency"` for Rp prefix
- Add `data-decimals="X"` for decimal places
- Add `data-submit-raw="true"` to submit raw values

✅ **Globally Available**
- All helpers included in every page
- No additional imports needed
- Works automatically on page load
