# Thousand Separator Implementation Guide

## Overview
A global JavaScript function has been implemented to automatically format numbers with thousand separators (Indonesian format: using dot `.` as thousand separator).

## How to Use

### 1. Basic Usage in Forms

Add these data attributes to any input field that needs thousand separator formatting:

```html
<!-- For regular numbers (no decimals) -->
<input type="text"
    name="quantity"
    data-format="number"
    data-submit-raw="true">

<!-- For numbers with decimals -->
<input type="text"
    name="distance"
    data-format="number"
    data-decimals="2"
    data-submit-raw="true">

<!-- For currency with Rp prefix -->
<input type="text"
    name="price"
    data-format="currency"
    data-submit-raw="true">
```

### 2. Data Attributes

| Attribute | Values | Description |
|-----------|--------|-------------|
| `data-format` | `number` or `currency` | Type of formatting to apply |
| `data-decimals` | Number (default: 0) | Number of decimal places |
| `data-submit-raw` | `true` or `false` | If true, submits raw number without formatting |

### 3. JavaScript Functions

#### Format a Number
```javascript
// Format without decimals: 1000 → "1.000"
formatNumber(1000);

// Format with 2 decimals: 1234.56 → "1.234,56"
formatNumber(1234.56, 2);
```

#### Parse a Formatted Number
```javascript
// Parse: "1.000" → 1000
parseNumber("1.000");

// Parse: "1.234,56" → 1234.56
parseNumber("1.234,56");
```

#### Format as Currency
```javascript
// Format: 1000 → "Rp 1.000"
formatCurrency(1000);
```

### 4. Dynamic Content

For dynamically added content (like table rows), reinitialize the formatter:

```javascript
// After adding new DOM elements
if (window.initNumberFormatting) {
    // Reinitialize for a specific container
    window.initNumberFormatting(containerElement);

    // Or reinitialize for entire document
    window.initNumberFormatting();
}
```

### 5. When to Use Thousand Separator

**Use thousand separator for:**
- Money amounts (prices, costs, salaries)
- Large quantities
- Distances (with decimals)
- Any numeric value that benefits from readability

**Don't use for:**
- Phone numbers
- ID numbers
- Codes (like NPWP, KTP)
- Years
- Small numbers (< 1000) where it's not beneficial

## Examples

### Price Input
```html
<div class="col-md-6">
    <label for="price">Price <span class="text-danger">*</span></label>
    <input type="text"
        class="form-control"
        id="price"
        name="price"
        value="{{ old('price', $model->price) }}"
        data-format="number"
        data-submit-raw="true"
        required>
</div>
```

### Distance Input (with decimals)
```html
<div class="col-md-4">
    <label for="distance">Distance (km)</label>
    <input type="text"
        class="form-control"
        id="distance"
        name="distance"
        value="{{ old('distance', $model->distance) }}"
        data-format="number"
        data-decimals="2"
        data-submit-raw="true">
</div>
```

### Money Input with Rp Prefix
```html
<div class="col-md-6">
    <label for="credit_limit">Credit Limit</label>
    <div class="input-group">
        <span class="input-group-text">Rp</span>
        <input type="text"
            class="form-control"
            id="credit_limit"
            name="credit_limit"
            value="{{ old('credit_limit', $model->credit_limit) }}"
            data-format="number"
            data-submit-raw="true">
    </div>
</div>
```

## Implementation Files

The thousand separator implementation is in:
- **JavaScript Helper**: `resources/assets/js/number-formatter.js`
- **Script Include**: `resources/views/layouts/sections/scripts.blade.php`

## Forms Updated

The following forms have been updated with thousand separator:
1. ✅ Price form (`resources/views/price/form.blade.php`)
2. ✅ Route form (`resources/views/route/form.blade.php`)
3. ✅ Customer form (`resources/views/customer/form.blade.php`)
4. ✅ Sales Order/Transaction form (`resources/views/transaction/sales-order/form.blade.php`)

## Notes

- The formatter automatically initializes on page load
- Formatting happens in real-time as users type
- Raw numeric values are submitted to the server (when `data-submit-raw="true"`)
- The formatter preserves cursor position during typing for better UX
- Indonesian format: `1.234.567,89` (dot for thousands, comma for decimals)
