# Validation Error Handling - Complete Guide

## Summary

✅ **YES! All form fields already have validation error display implemented.**

Every form in the application follows the same consistent pattern for showing validation errors from Laravel Form Requests.

## Implementation Pattern

All forms use this standard Laravel validation error display pattern:

### 1. Input Fields with Error Styling

```blade
<input type="text"
       class="form-control @error('field_name') is-invalid @enderror"
       id="field_name"
       name="field_name"
       value="{{ old('field_name', $model->field_name) }}"
       required>
@error('field_name')
    <div class="invalid-feedback">{{ $message }}</div>
@enderror
```

**How it works:**
- `@error('field_name') is-invalid @enderror` - Adds red border when validation fails
- `old('field_name', $model->field_name)` - Preserves user input on validation failure
- `@error('field_name')...@enderror` - Displays the actual error message below the field
- `.invalid-feedback` - Bootstrap class that shows the error message in red

### 2. Select Fields with Error Styling

```blade
<select class="form-select @error('field_name') is-invalid @enderror"
        id="field_name"
        name="field_name"
        required>
    <option value="">-- Select Option --</option>
    @foreach($options as $option)
        <option value="{{ $option->id }}"
                {{ old('field_name', $model->field_name) == $option->id ? 'selected' : '' }}>
            {{ $option->name }}
        </option>
    @endforeach
</select>
@error('field_name')
    <div class="invalid-feedback">{{ $message }}</div>
@enderror
```

### 3. Checkbox Fields with Error Styling

```blade
<div class="form-check">
    <input class="form-check-input @error('is_active') is-invalid @enderror"
           type="checkbox"
           id="is_active"
           name="is_active"
           value="1"
           {{ old('is_active', $model->is_active) ? 'checked' : '' }}>
    <label class="form-check-label" for="is_active">
        Active
    </label>
</div>
@error('is_active')
    <div class="invalid-feedback d-block">{{ $message }}</div>
@enderror
```

**Note:** For checkboxes, use `d-block` class to make error visible since checkboxes don't have default invalid feedback display.

### 4. Input Group Fields (with prefix/suffix)

```blade
<div class="input-group">
    <span class="input-group-text">Rp</span>
    <input type="number"
           class="form-control @error('credit_limit') is-invalid @enderror"
           id="credit_limit"
           name="credit_limit"
           value="{{ old('credit_limit', $model->credit_limit) }}"
           min="0">
    @error('credit_limit')
        <div class="invalid-feedback">{{ $message }}</div>
    @enderror
</div>
```

## Confirmed Forms with Validation Errors ✅

All the following forms have complete validation error handling:

1. ✅ **Customer Form** (`resources/views/customer/form.blade.php`)
   - Full name, email, phone, address
   - PIC name, PIC phone
   - Payment terms, credit limit
   - Tax checkboxes (PPN, PPh)
   - NPWP number

2. ✅ **Truck Form** (`resources/views/truck/form.blade.php`)
   - Truck type, code, brand
   - Body number, machine number
   - License plate, vehicle year
   - All document fields (STNK, KIR, BPKB)
   - Credit dates, GPS info

3. ✅ **Truck Type Form** (`resources/views/truck-type/form.blade.php`)
   - Name field

4. ✅ **Role Form** (`resources/views/role/form.blade.php`)
   - Role name
   - Permissions array

5. ✅ **Driver Form** (`resources/views/driver/form.blade.php`)
   - Code, full name, nickname
   - And all other driver fields

6. ✅ **Location Form** (`resources/views/location/form.blade.php`)

7. ✅ **Route Form** (`resources/views/route/form.blade.php`)

8. ✅ **Price Form** (`resources/views/price/form.blade.php`)

9. ✅ **Truck Loadout Form** (`resources/views/truck-loadout/form.blade.php`)

10. ✅ **Account Forms** (Type, Category, Account)

11. ✅ **User Form** (`resources/views/user/form.blade.php`)

12. ✅ **Journal Form** (`resources/views/journal/form.blade.php`)

## How Validation Works End-to-End

### 1. User Submits Form
```html
<form method="POST" action="{{ route('customer.store') }}">
    @csrf
    <input name="email" value="invalid-email">
    <button type="submit">Submit</button>
</form>
```

### 2. Controller Uses Form Request
```php
public function store(CustomerRequest $request)
{
    // If validation fails, user is redirected back
    // If validation passes, code continues
    $this->customerRepository->create($request->validated());

    return redirect()
        ->route('customer.index')
        ->with('success', 'Customer created successfully.');
}
```

### 3. Form Request Validates Data
```php
class CustomerRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'email' => 'required|email|unique:customers,email',
            'full_name' => 'required|string|max:255',
            // ... more rules
        ];
    }

    public function messages(): array
    {
        return [
            'email.required' => 'The email address field is required.',
            'email.email' => 'Please enter a valid email address.',
            'email.unique' => 'A customer with this email already exists.',
        ];
    }
}
```

### 4. Form Shows Errors
If validation fails, Laravel automatically:
- Redirects back to the form
- Stores errors in session
- Preserves old input values
- Makes `$errors` variable available in view

The form then displays:
```html
<!-- Input has red border -->
<input type="email" class="form-control is-invalid"
       name="email" value="invalid-email">

<!-- Error message shows below -->
<div class="invalid-feedback">Please enter a valid email address.</div>
```

## Visual Example

When validation fails:

```
┌─────────────────────────────────────────┐
│ Email Address *                         │
│ ┌─────────────────────────────────────┐ │ ← Red border
│ │ invalid-email                       │ │
│ └─────────────────────────────────────┘ │
│ ⚠ Please enter a valid email address.  │ ← Error message in red
└─────────────────────────────────────────┘
```

When validation passes:
```
┌─────────────────────────────────────────┐
│ Email Address *                         │
│ ┌─────────────────────────────────────┐ │ ← Normal border
│ │ valid@email.com                     │ │
│ └─────────────────────────────────────┘ │
└─────────────────────────────────────────┘
```

## Combined with Toast Notifications

The application provides **two levels of feedback**:

### 1. **Inline Field Errors** (Validation)
- Show **specific field errors** from Form Request validation
- Appear **below each field** with red border
- Help user **fix individual field issues**
- Example: "Please enter a valid email address."

### 2. **Toast Notifications** (Success/Error)
- Show **overall operation result** at **top-right**
- Appear after **successful submission** or **critical failures**
- Provide **general feedback** about the action
- Example: "Customer created successfully!" or "Failed to create customer: Database error"

### How They Work Together

**Scenario 1: Validation Fails**
```
User submits form → Validation fails → Redirected back to form
                                     → Inline errors show below fields
                                     → OLD input values preserved
                                     → NO toast notification
```

**Scenario 2: Validation Passes, Save Succeeds**
```
User submits form → Validation passes → Data saved to database
                                      → Redirected to index page
                                      → Toast shows: "Customer created successfully!"
```

**Scenario 3: Validation Passes, Save Fails**
```
User submits form → Validation passes → Database error occurs
                                      → Redirected back to form
                                      → OLD input values preserved
                                      → Toast shows: "Failed to create customer: [error]"
```

## Best Practices Already Implemented ✅

1. ✅ **Consistent Pattern**: All forms use the same error display pattern
2. ✅ **User-Friendly Messages**: Custom error messages in Form Requests
3. ✅ **Preserve Input**: `old()` helper preserves user input on failure
4. ✅ **Visual Feedback**: Red borders and error text make errors obvious
5. ✅ **Accessibility**: Error messages are properly associated with fields
6. ✅ **Bootstrap Integration**: Uses Bootstrap's `.is-invalid` and `.invalid-feedback` classes
7. ✅ **Specific Error Messages**: Each field shows its own specific error
8. ✅ **Required Field Indicators**: Asterisks (*) mark required fields

## Testing Validation

To test validation error display:

1. **Try submitting empty form** - Required field errors should appear
2. **Enter invalid email** - Email format error should show
3. **Enter duplicate value** - Unique constraint error should show
4. **Enter text in number field** - Type validation error should show
5. **All errors should persist** - After submission, form should show all errors
6. **Input should be preserved** - Your entered values should remain (except invalid ones)

## Conclusion

✅ **All forms in your application already have complete validation error handling!**

No additional work is needed. The combination of:
- Laravel Form Requests (backend validation)
- Blade error directives (`@error`)
- Bootstrap invalid feedback classes
- Toast notifications (for success/error)

...provides a **complete, user-friendly validation experience** across the entire application.
