<?php

namespace App\Console\Commands;

use Error;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use PhpParser\Node\Expr\Throw_;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Throwable;

class GeneratePermission extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:generate-permission';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $resources = config('permissions.resources');
        $customPermissions = config('permissions.custom_permissions');
        $excludedPermissions = config('permissions.excluded_permissions', []);

        $role = Role::findById(1);

        $this->info('Generating permission...');

        try {
            // Generate CRUD permissions for each resource
            foreach ($resources as $resource => $label) {
                $actions = ['index', 'create', 'store', 'show', 'edit', 'update', 'destroy', 'data'];

                foreach ($actions as $action) {
                    $permissionName = "$resource.$action";

                    // Check if permission is excluded
                    if ($this->isExcluded($permissionName, $excludedPermissions)) {
                        $this->info("Skipped excluded permission: $permissionName");
                        continue;
                    }

                    try {
                        $permission = Permission::findByName($permissionName);
                    } catch (Exception | Error $e) {
                        $permission = null;
                    }

                    if (!$permission) {
                        $permission = Permission::create(["name" => $permissionName]);
                        $role->givePermissionTo($permission);
                        $this->info("Created permission: $permissionName");
                    } else {
                        $this->info("Permission already exists: $permissionName");
                    }
                }
            }

            // Generate custom permissions
            foreach ($customPermissions as $resource => $permissions) {
                foreach ($permissions as $action => $label) {
                    $permissionName = "$resource.$action";

                    // Check if permission is excluded
                    if ($this->isExcluded($permissionName, $excludedPermissions)) {
                        $this->info("Skipped excluded permission: $permissionName");
                        continue;
                    }

                    try {
                        $permission = Permission::findByName($permissionName);
                    } catch (Exception | Error $e) {
                        $permission = null;
                    }

                    if (!$permission) {
                        $permission = Permission::create(["name" => $permissionName]);
                        $role->givePermissionTo($permission);
                        $this->info("Created permission: $permissionName");
                    } else {
                        $this->info("Permission already exists: $permissionName");
                    }
                }
            }

            $this->info('✓ Permission generation completed!');
        } catch (Exception | Error  $e) {
            Log::error('Console Error: ' . $e);

            $this->error('Failed to create permission: ' . $e->getMessage());
        }
    }

    /**
     * Check if a permission is excluded based on exclusion patterns
     */
    private function isExcluded(string $permission, array $excludedPermissions): bool
    {
        foreach ($excludedPermissions as $pattern) {
            // Handle wildcard patterns like '*.data' or 'customer.*'
            // Escape dots first, then replace asterisks with .*
            $regex = '/^' . str_replace('*', '.*', str_replace('.', '\.', $pattern)) . '$/';
            if (preg_match($regex, $permission)) {
                return true;
            }
        }
        return false;
    }
}
