<?php

namespace App\Helpers;

use Spatie\Activitylog\Models\Activity;
use Illuminate\Support\Collection;

class ActivityLogHelper
{
    /**
     * Get formatted activity logs for a model.
     */
    public static function getActivitiesForModel($model, int $limit = 50): Collection
    {
        return Activity::where('subject_type', get_class($model))
            ->where('subject_id', $model->id)
            ->with('causer')
            ->latest()
            ->limit($limit)
            ->get()
            ->map(function ($activity) {
                return self::formatActivity($activity);
            });
    }

    /**
     * Get all activity logs with pagination support.
     */
    public static function getAllActivities(int $perPage = 50)
    {
        return Activity::with(['causer', 'subject'])
            ->latest()
            ->paginate($perPage);
    }

    /**
     * Get activities by causer (user who made the change).
     */
    public static function getActivitiesByCauser($causerId, int $limit = 50): Collection
    {
        return Activity::where('causer_id', $causerId)
            ->with('subject')
            ->latest()
            ->limit($limit)
            ->get()
            ->map(function ($activity) {
                return self::formatActivity($activity);
            });
    }

    /**
     * Format activity log for display.
     */
    public static function formatActivity(Activity $activity): array
    {
        $formatted = [
            'id' => $activity->id,
            'description' => $activity->description,
            'event' => $activity->event,
            'created_at' => $activity->created_at->format('Y-m-d H:i:s'),
            'causer' => $activity->causer ? $activity->causer->name : 'System',
            'subject_type' => class_basename($activity->subject_type),
            'subject_id' => $activity->subject_id,
        ];

        // Add property changes if available
        if ($activity->properties) {
            $formatted['changes'] = self::formatChanges($activity);
        }

        return $formatted;
    }

    /**
     * Format changes for better readability.
     */
    protected static function formatChanges(Activity $activity): array
    {
        $changes = [];

        // Fields to exclude from all logs (timestamps, audit fields, calculated fields, and IDs)
        $excludeFields = [
            'created_at', 'updated_at', 'deleted_at',
            'created_by', 'updated_by', 'deleted_by',
            // Calculated fields that should not be shown
            'sub_total', 'total', 'grand_total', 'ppn', 'pph',
            // ID fields that should not be shown
            'id'
        ];

        // Handle attribute changes (from LogsActivity trait)
        if ($activity->properties->has('attributes') && $activity->properties->has('old')) {
            $new = $activity->properties->get('attributes');
            $old = $activity->properties->get('old');

            // Try to get the subject model for attribute labels
            $subject = $activity->subject;

            foreach ($new as $key => $value) {
                // Skip excluded fields
                if (in_array($key, $excludeFields)) {
                    continue;
                }

                if (!isset($old[$key]) || $old[$key] != $value) {
                    $label = $key;
                    $oldReadable = $old[$key] ?? null;
                    $newReadable = $value;

                    // Get human-readable labels and values if subject exists
                    if ($subject && method_exists($subject, 'getAttributeLabel')) {
                        $label = $subject->getAttributeLabel($key);
                    }

                    if ($subject && method_exists($subject, 'getReadableAttributeValue')) {
                        $oldReadable = $subject->getReadableAttributeValue($key, $old[$key] ?? null);
                        $newReadable = $subject->getReadableAttributeValue($key, $value);
                    }

                    $changes['attributes'][$key] = [
                        'label' => $label,
                        'old' => $old[$key] ?? null,
                        'new' => $value,
                        'old_readable' => $oldReadable,
                        'new_readable' => $newReadable,
                    ];
                }
            }
        }

        // Handle detail changes (from ModelWithDetailsObserver)
        if ($activity->properties->has('detail_changes')) {
            $detailChanges = $activity->properties->get('detail_changes');

            if (!empty($detailChanges['added'])) {
                $changes['details_added'] = self::formatDetailChanges($detailChanges['added'], 'added', $activity);
            }

            if (!empty($detailChanges['updated'])) {
                $changes['details_updated'] = self::formatDetailChanges($detailChanges['updated'], 'updated', $activity);
            }

            if (!empty($detailChanges['deleted'])) {
                $changes['details_deleted'] = self::formatDetailChanges($detailChanges['deleted'], 'deleted', $activity);
            }
        }

        // Handle permission changes (from RoleRepository)
        if ($activity->properties->has('permission_changes')) {
            $permissionChanges = $activity->properties->get('permission_changes');

            if (!empty($permissionChanges['added'])) {
                $changes['permissions_added'] = self::formatPermissionNames($permissionChanges['added']);
                // Include the grouped count for display
                if (isset($permissionChanges['added_count'])) {
                    $changes['permissions_added_count'] = $permissionChanges['added_count'];
                }
            }

            if (!empty($permissionChanges['removed'])) {
                $changes['permissions_removed'] = self::formatPermissionNames($permissionChanges['removed']);
                // Include the grouped count for display
                if (isset($permissionChanges['removed_count'])) {
                    $changes['permissions_removed_count'] = $permissionChanges['removed_count'];
                }
            }
        }

        return $changes;
    }

    /**
     * Format detail changes with human-readable labels.
     */
    protected static function formatDetailChanges(array $details, string $type, Activity $activity): array
    {
        $formatted = [];
        $detailModelClass = self::getDetailModelClass($activity->subject_type);

        // Fields to exclude from all logs (timestamps, audit fields, calculated fields, and IDs)
        $excludeFields = [
            'created_at', 'updated_at', 'deleted_at',
            'created_by', 'updated_by', 'deleted_by',
            // Calculated fields that should not be shown
            'sub_total', 'total', 'grand_total', 'ppn', 'pph',
            // ID fields that should not be shown
            'id'
        ];

        foreach ($details as $detail) {
            $formattedDetail = [
                'fields' => []
            ];

            if ($type === 'updated') {
                // For updates, we have 'old', 'new', and 'changes' keys
                $changes = $detail['changes'] ?? [];

                foreach ($changes as $field => $newValue) {
                    // Skip excluded fields
                    if (in_array($field, $excludeFields)) {
                        continue;
                    }

                    $oldValue = $detail['old'][$field] ?? null;

                    $formattedDetail['fields'][] = [
                        'field' => $field,
                        'label' => self::getDetailFieldLabel($detailModelClass, $field),
                        'old_value' => $oldValue,
                        'new_value' => $newValue,
                        'old_readable' => self::formatDetailValue($detailModelClass, $field, $oldValue),
                        'new_readable' => self::formatDetailValue($detailModelClass, $field, $newValue),
                    ];
                }
            } else {
                // For added/deleted, show all fields
                foreach ($detail as $field => $value) {
                    // Skip excluded fields
                    if (in_array($field, $excludeFields)) {
                        continue;
                    }

                    $formattedDetail['fields'][] = [
                        'field' => $field,
                        'label' => self::getDetailFieldLabel($detailModelClass, $field),
                        'value' => $value,
                        'value_readable' => self::formatDetailValue($detailModelClass, $field, $value),
                    ];
                }
            }

            $formatted[] = $formattedDetail;
        }

        return $formatted;
    }

    /**
     * Get detail model class from subject type.
     */
    protected static function getDetailModelClass(string $subjectType): string
    {
        $modelName = class_basename($subjectType);

        return match($modelName) {
            'Transaction' => \App\Models\TransactionDetail::class,
            'Journal' => \App\Models\AccountJournal::class,
            'Invoice' => \App\Models\InvoiceDetail::class,
            'DeliveryOrder' => \App\Models\DeliverOrderDetail::class,
            default => $subjectType . 'Detail',
        };
    }

    /**
     * Get human-readable label for a detail field.
     */
    protected static function getDetailFieldLabel(string $modelClass, string $field): string
    {
        if (class_exists(\App\Models\ModelAttributeLabels::class)) {
            $labels = \App\Models\ModelAttributeLabels::getLabelsFor($modelClass);
            if (isset($labels[$field])) {
                return $labels[$field];
            }
        }

        return ucwords(str_replace('_', ' ', $field));
    }

    /**
     * Format detail value to be human-readable.
     */
    protected static function formatDetailValue(string $modelClass, string $field, $value): string
    {
        if ($value === null) {
            return 'N/A';
        }

        // Handle foreign keys - try to get related model name
        if (preg_match('/^id_(.+)/', $field, $matches)) {
            $relationName = \Illuminate\Support\Str::camel($matches[1]);
            $relatedModelClass = self::getRelatedModelClass($modelClass, $relationName);

            if ($relatedModelClass && class_exists($relatedModelClass)) {
                try {
                    $related = $relatedModelClass::find($value);
                    if ($related) {
                        if (isset($related->name)) {
                            return $related->name;
                        }
                        if (isset($related->code)) {
                            return $related->code;
                        }
                        if (isset($related->full_name)) {
                            return $related->full_name;
                        }
                    }
                } catch (\Exception $e) {
                    // Fallback to ID
                }
            }

            return "ID: {$value}";
        }

        // Handle currency fields
        if (in_array($field, ['price', 'sub_total', 'total', 'grand_total', 'pocket_money_1', 'pocket_money_2', 'pocket_money_3', 'bonus', 'discount', 'ppn', 'pph'])) {
            return \App\Helpers\Helpers::formatCurrency($value, 2);
        }

        return (string) $value;
    }

    /**
     * Get related model class from relation name.
     */
    protected static function getRelatedModelClass(string $modelClass, string $relationName): ?string
    {
        $relations = [
            'origin' => \App\Models\Location::class,
            'destination' => \App\Models\Location::class,
            'loadout' => \App\Models\TruckLoadout::class,
            'transaction' => \App\Models\Transaction::class,
            'customer' => \App\Models\Customer::class,
            'sales' => \App\Models\User::class,
            'driver' => \App\Models\Driver::class,
            'truck' => \App\Models\Truck::class,
            'account' => \App\Models\Account::class,
            'accounts' => \App\Models\Account::class,
            'journal' => \App\Models\Journal::class,
        ];

        return $relations[$relationName] ?? null;
    }

    /**
     * Get human-readable change summary.
     */
    public static function getChangeSummary(Activity $activity): string
    {
        $parts = [];

        // Fields to exclude from all logs (timestamps, audit fields, calculated fields, and IDs)
        $excludeFields = [
            'created_at', 'updated_at', 'deleted_at',
            'created_by', 'updated_by', 'deleted_by',
            // Calculated fields that should not be shown
            'sub_total', 'total', 'grand_total', 'ppn', 'pph',
            // ID fields that should not be shown
            'id'
        ];

        if ($activity->properties->has('attributes') && $activity->properties->has('old')) {
            $new = $activity->properties->get('attributes');
            $old = $activity->properties->get('old');
            $changedFields = [];
            $subject = $activity->subject;

            foreach ($new as $key => $value) {
                // Skip excluded fields
                if (in_array($key, $excludeFields)) {
                    continue;
                }

                if (!isset($old[$key]) || $old[$key] != $value) {
                    // Get human-readable label if possible
                    if ($subject && method_exists($subject, 'getAttributeLabel')) {
                        $changedFields[] = $subject->getAttributeLabel($key);
                    } else {
                        $changedFields[] = ucwords(str_replace('_', ' ', $key));
                    }
                }
            }

            if (!empty($changedFields)) {
                $parts[] = 'Changed: ' . implode(', ', $changedFields);
            }
        }

        if ($activity->properties->has('detail_changes')) {
            $detailChanges = $activity->properties->get('detail_changes');

            if (!empty($detailChanges['added'])) {
                $parts[] = count($detailChanges['added']) . ' detail(s) added';
            }

            if (!empty($detailChanges['updated'])) {
                $parts[] = count($detailChanges['updated']) . ' detail(s) updated';
            }

            if (!empty($detailChanges['deleted'])) {
                $parts[] = count($detailChanges['deleted']) . ' detail(s) deleted';
            }
        }

        if ($activity->properties->has('permission_changes')) {
            $permissionChanges = $activity->properties->get('permission_changes');

            if (!empty($permissionChanges['added'])) {
                // Use grouped count if available, otherwise count all permissions
                $count = $permissionChanges['added_count'] ?? count($permissionChanges['added']);
                $parts[] = $count . ' permission(s) added';
            }

            if (!empty($permissionChanges['removed'])) {
                // Use grouped count if available, otherwise count all permissions
                $count = $permissionChanges['removed_count'] ?? count($permissionChanges['removed']);
                $parts[] = $count . ' permission(s) removed';
            }
        }

        return implode(' | ', $parts) ?: 'No changes recorded';
    }

    /**
     * Get activity logs filtered by date range.
     */
    public static function getActivitiesByDateRange($startDate, $endDate, int $limit = 100): Collection
    {
        return Activity::whereBetween('created_at', [$startDate, $endDate])
            ->with(['causer', 'subject'])
            ->latest()
            ->limit($limit)
            ->get()
            ->map(function ($activity) {
                return self::formatActivity($activity);
            });
    }

    /**
     * Get activities for specific model type.
     */
    public static function getActivitiesByModelType(string $modelClass, int $limit = 50): Collection
    {
        return Activity::where('subject_type', $modelClass)
            ->with('causer')
            ->latest()
            ->limit($limit)
            ->get()
            ->map(function ($activity) {
                return self::formatActivity($activity);
            });
    }

    /**
     * Format permission names to human-readable format.
     * Converts 'customer.create' to 'Customer - Create'
     * Groups related permissions (create+store, edit+update)
     */
    protected static function formatPermissionNames(array $permissions): array
    {
        $resources = config('permissions.resources', []);
        $customs = config('permissions.custom_permissions', []);
        $actions = [
            'index' => 'View All',
            'show' => 'View Detail',
            'create' => 'Create',
            'store' => 'Store',
            'edit' => 'Edit',
            'update' => 'Update',
            'destroy' => 'Delete',
        ];

        // Group related permissions
        $grouped = [];
        $processed = [];
        $relatedActions = [
            'create' => 'store',
            'store' => 'create',
            'edit' => 'update',
            'update' => 'edit',
        ];

        foreach ($permissions as $permission) {
            if (in_array($permission, $processed)) {
                continue;
            }

            [$resource, $action] = explode('.', $permission) + [null, null];

            if (!$resource || !$action) {
                $grouped[] = ucwords(str_replace(['.', '-', '_'], ' ', $permission));
                $processed[] = $permission;
                continue;
            }

            $resourceLabel = $resources[$resource] ?? ucwords(str_replace('-', ' ', $resource));

            // Check if this action has a related action
            if (isset($relatedActions[$action])) {
                $relatedAction = $relatedActions[$action];
                $relatedPermission = "{$resource}.{$relatedAction}";

                // If the related permission is also in the list, group them
                if (in_array($relatedPermission, $permissions)) {
                    // Use the primary action label (create for create+store, update for edit+update)
                    $primaryAction = in_array($action, ['create', 'update']) ? $action : $relatedAction;
                    $actionLabel = $customs[$resource][$primaryAction] ?? $actions[$primaryAction] ?? ucwords(str_replace('-', ' ', $primaryAction));

                    $grouped[] = "{$resourceLabel} - {$actionLabel}";
                    $processed[] = $permission;
                    $processed[] = $relatedPermission;
                    continue;
                }
            }

            // No grouping needed, format individually
            $actionLabel = $customs[$resource][$action] ?? $actions[$action] ?? ucwords(str_replace('-', ' ', $action));
            $grouped[] = "{$resourceLabel} - {$actionLabel}";
            $processed[] = $permission;
        }

        return $grouped;
    }
}
