<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Str;

class Helpers
{
  /**
   * Generate menu attributes for semi-dark mode
   *
   * @param bool $semiDarkEnabled Whether semi-dark mode is enabled
   * @return array HTML attributes for the menu element
   */
  public static function getMenuAttributes($semiDarkEnabled)
  {
    $attributes = [];

    if ($semiDarkEnabled) {
      $attributes['data-bs-theme'] = 'dark';
    }

    return $attributes;
  }

  public static function appClasses()
  {

    $data = config('custom.custom');

    // default data array
    $DefaultData = [
      'myLayout' => 'vertical',
      'myTheme' => 'light',
      'mySkins' => 'default',
      'hasSemiDark' => false,
      'myRTLMode' => true,
      'hasCustomizer' => true,
      'showDropdownOnHover' => true,
      'displayCustomizer' => true,
      'contentLayout' => 'compact',
      'headerType' => 'fixed',
      'navbarType' => 'sticky',
      'menuFixed' => true,
      'menuCollapsed' => false,
      'footerFixed' => false,
      'customizerControls' => [
        'color',
        'theme',
        'skins',
        'semiDark',
        'layoutCollapsed',
        'layoutNavbarOptions',
        'headerType',
        'contentLayout',
        'rtl'
      ],
      //   'defaultLanguage'=>'en',
    ];

    // if any key missing of array from custom.php file it will be merge and set a default value from dataDefault array and store in data variable
    $data = array_merge($DefaultData, $data);

    // All options available in the template
    $allOptions = [
      'myLayout' => ['vertical', 'horizontal', 'blank', 'front'],
      'menuCollapsed' => [true, false],
      'hasCustomizer' => [true, false],
      'showDropdownOnHover' => [true, false],
      'displayCustomizer' => [true, false],
      'contentLayout' => ['compact', 'wide'],
      'headerType' => ['fixed', 'static'],
      'navbarType' => ['sticky', 'static', 'hidden'],
      'myTheme' => ['light', 'dark', 'system'],
      'mySkins' => ['default', 'bordered', 'raspberry'],
      'hasSemiDark' => [true, false],
      'myRTLMode' => [true, false],
      'menuFixed' => [true, false],
      'footerFixed' => [true, false],
      'customizerControls' => [],
      // 'defaultLanguage'=>array('en'=>'en','fr'=>'fr','de'=>'de','ar'=>'ar'),
    ];

    //if myLayout value empty or not match with default options in custom.php config file then set a default value
    foreach ($allOptions as $key => $value) {
      if (array_key_exists($key, $DefaultData)) {
        if (gettype($DefaultData[$key]) === gettype($data[$key])) {
          // data key should be string
          if (is_string($data[$key])) {
            // data key should not be empty
            if (isset($data[$key]) && $data[$key] !== null) {
              // data key should not be exist inside allOptions array's sub array
              if (!array_key_exists($data[$key], $value)) {
                // ensure that passed value should be match with any of allOptions array value
                $result = array_search($data[$key], $value, 'strict');
                if (empty($result) && $result !== 0) {
                  $data[$key] = $DefaultData[$key];
                }
              }
            } else {
              // if data key not set or
              $data[$key] = $DefaultData[$key];
            }
          }
        } else {
          $data[$key] = $DefaultData[$key];
        }
      }
    }
    $themeVal = $data['myTheme'] == "dark" ? "dark" : "light";
    $themeUpdatedVal = $data['myTheme'] == "dark" ? "dark" : $data['myTheme'];

    // Determine if the layout is admin or front based on template name
    $layoutName = $data['myLayout'];
    $isAdmin = !Str::contains($layoutName, 'front');

    $modeCookieName = $isAdmin ? 'admin-mode' : 'front-mode';
    $colorPrefCookieName = $isAdmin ? 'admin-colorPref' : 'front-colorPref';
    $primaryColorCookieName = $isAdmin ? 'admin-primaryColor' : 'front-primaryColor';

    // Get primary color from custom.php if explicitly set
    $primaryColor = null;
    if (array_key_exists('primaryColor', $data)) {
      $primaryColor = $data['primaryColor'];
    }

    // Check for primary color in cookie
    if (isset($_COOKIE[$primaryColorCookieName])) {
      $primaryColor = $_COOKIE[$primaryColorCookieName];
    }

    // Determine style based on cookies, only if not 'blank-layout'
    if ($layoutName !== 'blank') {
      if (isset($_COOKIE[$modeCookieName])) {
        $themeVal = $_COOKIE[$modeCookieName];
        if ($themeVal === 'system') {
          $themeVal = isset($_COOKIE[$colorPrefCookieName]) ? $_COOKIE[$colorPrefCookieName] : 'light';
        }
        $themeUpdatedVal = $_COOKIE[$modeCookieName];
      }
    }

    // Define standardized cookie names
    $skinCookieName = 'customize_skin';
    $semiDarkCookieName = 'customize_semi_dark';

    // Process skin and semi-dark settings only for admin layouts
    if ($isAdmin) {
      // Get skin from cookie or fall back to config
      $skinFromCookie = isset($_COOKIE[$skinCookieName]) ? $_COOKIE[$skinCookieName] : null;
      $configSkin = isset($data['mySkins']) ? $data['mySkins'] : 'default';
      $skinName = $skinFromCookie ?: $configSkin;

      // Get semi-dark setting from cookie or fall back to config
      $semiDarkFromCookie = isset($_COOKIE[$semiDarkCookieName]) ? $_COOKIE[$semiDarkCookieName] : null;
      // Ensure we have a proper boolean conversion
      $semiDarkEnabled = $semiDarkFromCookie !== null ?
        filter_var($semiDarkFromCookie, FILTER_VALIDATE_BOOLEAN) :
        (bool)$data['hasSemiDark'];
    } else {
      // For front-end layouts, use defaults
      $skinName = 'default';
      $semiDarkEnabled = false;
    }

    // Get menu Collapsed state from cookie or fall back to config
    $menuCollapsedFromCookie = isset($_COOKIE['LayoutCollapsed']) ? $_COOKIE['LayoutCollapsed'] : $data['menuCollapsed'];

    // Get content layout from cookie or fall back to config
    $contentLayoutFromCookie = isset($_COOKIE['contentLayout']) ? $_COOKIE['contentLayout'] : $data['contentLayout'];

    // Get header type from cookie or fall back to config
    $navbarTypeFromCookie = isset($_COOKIE['navbarType']) ? $_COOKIE['navbarType'] : $data['navbarType'];

    // Get Header type from cookie or fall back to config
    $headerTypeFromCookie = isset($_COOKIE['headerType']) ? $_COOKIE['headerType'] : $data['headerType'];

    $directionVal = isset($_COOKIE['direction']) ? ($_COOKIE['direction'] === 'true' ? 'rtl' : 'ltr') : $data['myRTLMode'];

    //layout classes
    $layoutClasses = [
      'layout' => $data['myLayout'],
      'skins' => $data['mySkins'],
      'skinName' => $skinName,
      'semiDark' => $semiDarkEnabled,
      'color' => $primaryColor,
      'theme' => $themeVal,
      'themeOpt' => $data['myTheme'],
      'themeOptVal' => $themeUpdatedVal,
      'rtlMode' => $data['myRTLMode'],
      'textDirection' => $directionVal,
      'menuCollapsed' => $menuCollapsedFromCookie,
      'hasCustomizer' => $data['hasCustomizer'],
      'showDropdownOnHover' => $data['showDropdownOnHover'],
      'displayCustomizer' => $data['displayCustomizer'],
      'contentLayout' => $contentLayoutFromCookie,
      'headerType' => $headerTypeFromCookie,
      'navbarType' => $navbarTypeFromCookie,
      'menuFixed' => $data['menuFixed'],
      'footerFixed' => $data['footerFixed'],
      'customizerControls' => $data['customizerControls'],
      'menuAttributes' => self::getMenuAttributes($semiDarkEnabled),
    ];

    // sidebar Collapsed
    if ($layoutClasses['menuCollapsed'] === 'true' || $layoutClasses['menuCollapsed'] === true) {
      $layoutClasses['menuCollapsed'] = 'layout-menu-collapsed';
    } else {
      $layoutClasses['menuCollapsed'] = '';
    }

    // Header Type
    if ($layoutClasses['headerType'] == 'fixed') {
      $layoutClasses['headerType'] = 'layout-menu-fixed';
    }
    // Navbar Type
    if ($layoutClasses['navbarType'] == 'sticky') {
      $layoutClasses['navbarType'] = 'layout-navbar-fixed';
    } elseif ($layoutClasses['navbarType'] == 'static') {
      $layoutClasses['navbarType'] = '';
    } else {
      $layoutClasses['navbarType'] = 'layout-navbar-hidden';
    }

    // Menu Fixed
    if ($layoutClasses['menuFixed'] == true) {
      $layoutClasses['menuFixed'] = 'layout-menu-fixed';
    }


    // Footer Fixed
    if ($layoutClasses['footerFixed'] == true) {
      $layoutClasses['footerFixed'] = 'layout-footer-fixed';
    }

    // RTL Layout/Mode
    if ($layoutClasses['rtlMode'] == true) {
      $layoutClasses['rtlMode'] = 'rtl';
      $layoutClasses['textDirection'] = isset($_COOKIE['direction']) ? ($_COOKIE['direction'] === 'true' ? 'rtl' : 'ltr') : 'rtl';
    } else {
      $layoutClasses['rtlMode'] = 'ltr';
      $layoutClasses['textDirection'] = isset($_COOKIE['direction']) && $_COOKIE['direction'] === 'true' ? 'rtl' : 'ltr';
    }

    // Show DropdownOnHover for Horizontal Menu
    if ($layoutClasses['showDropdownOnHover'] == true) {
      $layoutClasses['showDropdownOnHover'] = true;
    } else {
      $layoutClasses['showDropdownOnHover'] = false;
    }

    // To hide/show display customizer UI, not js
    if ($layoutClasses['displayCustomizer'] == true) {
      $layoutClasses['displayCustomizer'] = true;
    } else {
      $layoutClasses['displayCustomizer'] = false;
    }

    return $layoutClasses;
  }

  public static function updatePageConfig($pageConfigs)
  {
    $demo = 'custom';
    if (isset($pageConfigs)) {
      if (count($pageConfigs) > 0) {
        foreach ($pageConfigs as $config => $val) {
          Config::set('custom.' . $demo . '.' . $config, $val);
        }
      }
    }
  }

  /**
   * Generate CSS for primary color
   *
   * @param string $color Hex color code for primary color
   * @return string CSS for primary color
   */
  public static function generatePrimaryColorCSS($color)
  {
    if (!$color) return '';

    // Check if the color actually came from a cookie or explicit configuration
    // Don't generate CSS if there's no specific need for a custom color
    $configColor = config('custom.custom.primaryColor', null);
    $isFromCookie = isset($_COOKIE['admin-primaryColor']) || isset($_COOKIE['front-primaryColor']);

    if (!$configColor && !$isFromCookie) return '';

    $r = hexdec(substr($color, 1, 2));
    $g = hexdec(substr($color, 3, 2));
    $b = hexdec(substr($color, 5, 2));

    // Calculate contrast color based on YIQ formula
    $yiq = (($r * 299) + ($g * 587) + ($b * 114)) / 1000;
    $contrastColor = ($yiq >= 150) ? '#000' : '#fff';

    return <<<CSS
:root, [data-bs-theme=light], [data-bs-theme=dark] {
  --bs-primary: {$color};
  --bs-primary-rgb: {$r}, {$g}, {$b};
  --bs-primary-bg-subtle: rgba({$r}, {$g}, {$b}, 0.1);
  --bs-primary-border-subtle: rgba({$r}, {$g}, {$b}, 0.3);
  --bs-primary-contrast: {$contrastColor};
}
CSS;
  }

  /**
   * Convert model collection to dropdown options
   *
   * @param \Illuminate\Support\Collection|\Illuminate\Database\Eloquent\Collection $data Collection of models
   * @param string $nameField Field to use as option label
   * @param string $valueField Field to use as option value
   * @return array Array of [value => label] for dropdown options
   */
  public static function toDropdownOptions($data, $nameField, $valueField)
  {
    return $data->pluck($nameField, $valueField)->toArray();
  }

  /**
   * Add thousand separators to a number for display
   * Uses Laravel's Number helper for locale-aware formatting
   *
   * @param float|int|string|null $value Number to format
   * @return string Formatted number with thousand separators
   *
   * @example
   * formatter(1000) => "1.000"
   * formatter(1000000) => "1.000.000"
   * formatter("6000000") => "6.000.000"
   * formatter(5000000.00) => "5.000.000"
   */
  public static function formatter($value)
  {
    if ($value === null || $value === '') {
      return '';
    }

    // Convert to number first (handles "5000000.00" correctly)
    $number = (float)$value;

    // Use Laravel's Number helper with Indonesian locale (id)
    return \Illuminate\Support\Number::format($number, locale: 'id');
  }

  /**
   * Remove thousand separators from formatted number
   * Simple parser - removes dots and commas
   *
   * @param string|null $formattedValue Number with thousand separators
   * @return float Clean number without formatting
   *
   * @example
   * parser("1.000") => 1000
   * parser("1.000.000") => 1000000
   * parser("6.000.000") => 6000000
   */
  public static function parser($formattedValue)
  {
    if ($formattedValue === null || $formattedValue === '') {
      return 0;
    }

    // Remove all separators (dots and commas)
    $cleanValue = str_replace(['.', ','], '', (string)$formattedValue);

    return (float)$cleanValue;
  }

  /**
   * Alias for formatter() - for backward compatibility
   * Note: $decimals parameter is ignored (kept for backward compatibility only)
   */
  public static function formatNumber($value, $decimals = 0)
  {
    return self::formatter($value);
  }

  /**
   * Alias for parser() - for backward compatibility
   */
  public static function parseNumber($formattedValue)
  {
    return self::parser($formattedValue);
  }

  /**
   * Format currency with Rp prefix and thousand separator
   * Note: $decimals parameter is ignored (kept for backward compatibility only)
   *
   * @param float|int|string|null $value Number to format
   * @param int $decimals Ignored - kept for backward compatibility
   * @return string Formatted currency string
   *
   * @example
   * formatCurrency(1000) => "Rp 1.000"
   * formatCurrency(1000000) => "Rp 1.000.000"
   */
  public static function formatCurrency($value, $decimals = 0)
  {
    return 'Rp ' . self::formatter($value);
  }

  public static function checkPermission($permission)
  {
    return Auth::user()->can($permission);
  }

  /**
   * Get attribute label for a model field
   *
   * @param string $modelClass Full model class name (e.g., 'App\Models\Truck')
   * @param string $attribute Field name (e.g., 'id_type')
   * @return string Human-readable label
   *
   * @example
   * getAttributeLabel('App\Models\Truck', 'id_type') => "Truck Type"
   * getAttributeLabel('App\Models\Transaction', 'id_customer') => "Customer"
   */
  public static function getAttributeLabel($modelClass, $attribute)
  {
    if (class_exists(\App\Models\ModelAttributeLabels::class)) {
      $labels = \App\Models\ModelAttributeLabels::getLabelsFor($modelClass);

      if (isset($labels[$attribute])) {
        return $labels[$attribute];
      }
    }

    // Fallback: convert snake_case to Title Case
    return ucwords(str_replace('_', ' ', $attribute));
  }

  /**
   * Get all attribute labels for a model
   *
   * @param string $modelClass Full model class name
   * @return array Associative array of attribute => label
   *
   * @example
   * getModelLabels('App\Models\Truck') => ['id_type' => 'Truck Type', 'code' => 'Code', ...]
   */
  public static function getModelLabels($modelClass)
  {
    if (class_exists(\App\Models\ModelAttributeLabels::class)) {
      return \App\Models\ModelAttributeLabels::getLabelsFor($modelClass);
    }

    return [];
  }

  /**
   * Get attribute label from model instance
   *
   * @param mixed $model Model instance
   * @param string $attribute Field name
   * @return string Human-readable label
   *
   * @example
   * $truck = Truck::find(1);
   * modelLabel($truck, 'id_type') => "Truck Type"
   */
  public static function modelLabel($model, $attribute)
  {
    if (is_object($model) && method_exists($model, 'getAttributeLabel')) {
      return $model->getAttributeLabel($attribute);
    }

    $modelClass = is_object($model) ? get_class($model) : $model;
    return self::getAttributeLabel($modelClass, $attribute);
  }

  /**
   * Make HTTP request to external API
   * Uses Laravel's HTTP client (Guzzle-based) for reliable API calls
   *
   * @param string $method HTTP method (GET, POST, PUT, PATCH, DELETE)
   * @param string $url Full URL to the API endpoint
   * @param array $data Request data (for POST/PUT/PATCH) or query params (for GET)
   * @param array $headers Additional headers
   * @param int $timeout Timeout in seconds (default: 30)
   * @return array Response array with 'success', 'status', 'data', and 'error' keys
   *
   * @example
   * // GET request
   * $result = Helpers::httpRequest('GET', 'https://api.example.com/users', ['id' => 123]);
   *
   * // POST request with JSON
   * $result = Helpers::httpRequest('POST', 'https://api.example.com/users', [
   *   'name' => 'John Doe',
   *   'email' => 'john@example.com'
   * ]);
   *
   * // POST with custom headers
   * $result = Helpers::httpRequest('POST', 'https://api.example.com/data', $data, [
   *   'Authorization' => 'Bearer token123',
   *   'X-Custom-Header' => 'value'
   * ]);
   *
   * // Check response
   * if ($result['success']) {
   *   $responseData = $result['data'];
   * } else {
   *   $errorMessage = $result['error'];
   * }
   */
  public static function httpRequest($method, $url, $data = [], $headers = [], $timeout = 30)
  {
    try {
      $method = strtoupper($method);

      // Create HTTP client with timeout
      $http = \Illuminate\Support\Facades\Http::timeout($timeout);

      // Add custom headers if provided
      if (!empty($headers)) {
        $http = $http->withHeaders($headers);
      }

      // Make request based on method
      $response = match($method) {
        'GET' => $http->get($url, $data),
        'POST' => $http->post($url, $data),
        'PUT' => $http->put($url, $data),
        'PATCH' => $http->patch($url, $data),
        'DELETE' => $http->delete($url, $data),
        default => throw new \InvalidArgumentException("Unsupported HTTP method: {$method}")
      };

      // Return structured response
      return [
        'success' => $response->successful(),
        'status' => $response->status(),
        'data' => $response->json(),
        'body' => $response->body(),
        'headers' => $response->headers(),
        'error' => $response->failed() ? $response->body() : null
      ];

    } catch (\Illuminate\Http\Client\ConnectionException $e) {
      // Connection timeout or network error
      return [
        'success' => false,
        'status' => 0,
        'data' => null,
        'body' => null,
        'headers' => [],
        'error' => 'Connection error: ' . $e->getMessage()
      ];

    } catch (\Exception $e) {
      // General error
      return [
        'success' => false,
        'status' => 0,
        'data' => null,
        'body' => null,
        'headers' => [],
        'error' => 'Request failed: ' . $e->getMessage()
      ];
    }
  }
}
