<?php

namespace App\Http\Controllers;

use App\Http\Requests\AccountCategoryRequest;
use App\Models\AccountCategory;
use App\Repositories\AccountCategoryRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class AccountCategoryController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $accountCategoryRepository;

    public function __construct(AccountCategoryRepository $accountCategoryRepository)
    {
        $this->accountCategoryRepository = $accountCategoryRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:account-category.index', only: ['index']),
            new Middleware('permission:account-category.show', only: ['show']),
            new Middleware('permission:account-category.store', only: ['create', 'store']),
            new Middleware('permission:account-category.update', only: ['edit', 'update']),
            new Middleware('permission:account-category.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 1,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "name", "title" => "Category", "orderable" => true, "className" => "text-start"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "account-category.show"],
            ["type" => "edit", "url" => "account-category.edit"],
            ["type" => "delete", "url" => "account-category.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "name",
                "label" => "Category",
                "col" => 6
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = AccountCategory::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'account-category');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("account-category.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new AccountCategory();

        return view("account-category.form", [
            "model" => $model,
            "isBackUrl" => route("account-category.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AccountCategoryRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->accountCategoryRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('account-category.index')
                ->with('success', 'Account category created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create account category');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->accountCategoryRepository->findOrFail($id);

        return view("account-category.show", [
            "model" => $model,
            "isBackUrl" => route("account-category.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->accountCategoryRepository->findOrFail($id);

        return view("account-category.form", [
            "model" => $model,
            "isBackUrl" => route("account-category.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AccountCategoryRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->accountCategoryRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('account-category.index')
                ->with('success', 'Account category updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update account category');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->accountCategoryRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Account category deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete account category');
        }
    }
}
