<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\AccountRequest;
use App\Models\Account;
use App\Models\AccountCategory;
use App\Models\AccountType;
use App\Repositories\AccountRepository;
use App\Repositories\AccountCategoryRepository;
use App\Repositories\AccountTypeRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class AccountController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $accountRepository;
    protected $accountCategoryRepository;
    protected $accountTypeRepository;

    public function __construct(
        AccountRepository $accountRepository,
        AccountCategoryRepository $accountCategoryRepository,
        AccountTypeRepository $accountTypeRepository
    ) {
        $this->accountRepository = $accountRepository;
        $this->accountCategoryRepository = $accountCategoryRepository;
        $this->accountTypeRepository = $accountTypeRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:account.index', only: ['index']),
            new Middleware('permission:account.show', only: ['show']),
            new Middleware('permission:account.store', only: ['create', 'store']),
            new Middleware('permission:account.update', only: ['edit', 'update']),
            new Middleware('permission:account.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 4,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "code", "title" => "Code", "orderable" => true, "className" => "text-start"],
            ["data" => "name", "title" => "Account Name", "orderable" => true, "className" => "text-start"],
            ["data" => "id_category", "title" => "Category", "orderable" => true, "relation" => "category", "relation_column" => "name", "className" => "text-start"],
            ["data" => "id_type", "title" => "Type", "orderable" => true, "relation" => "type", "relation_column" => "name", "className" => "text-start"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "account.show"],
            ["type" => "edit", "url" => "account.edit"],
            ["type" => "delete", "url" => "account.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "code",
                "label" => "Code",
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "name",
                "label" => "Account Name",
                "col" => 3
            ],
            [
                "type" => "select",
                "name" => "id_category",
                "label" => "Category",
                "options" => Helpers::toDropdownOptions(AccountCategory::all(), "name", "id"),
                "col" => 3
            ],
            [
                "type" => "select",
                "name" => "id_type",
                "label" => "Type",
                "options" => Helpers::toDropdownOptions(AccountType::all(), "name", "id"),
                "col" => 3
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Account::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'account');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("account.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Account();
        $categories = $this->accountCategoryRepository->all();
        $types = $this->accountTypeRepository->all();

        return view("account.form", [
            "model" => $model,
            "categories" => $categories,
            "types" => $types,
            "isBackUrl" => route("account.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AccountRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->accountRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('account.index')
                ->with('success', 'Account created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create account');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->accountRepository->findOrFail($id);

        return view("account.show", [
            "model" => $model,
            "isBackUrl" => route("account.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->accountRepository->findOrFail($id);
        $categories = $this->accountCategoryRepository->all();
        $types = $this->accountTypeRepository->all();

        return view("account.form", [
            "model" => $model,
            "categories" => $categories,
            "types" => $types,
            "isBackUrl" => route("account.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AccountRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->accountRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('account.index')
                ->with('success', 'Account updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update account');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->accountRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Account deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete account');
        }
    }
}
