<?php

namespace App\Http\Controllers;

use App\Http\Requests\AccountTypeRequest;
use App\Models\AccountType;
use App\Repositories\AccountTypeRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class AccountTypeController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $accountTypeRepository;

    public function __construct(AccountTypeRepository $accountTypeRepository)
    {
        $this->accountTypeRepository = $accountTypeRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:account-type.index', only: ['index']),
            new Middleware('permission:account-type.show', only: ['show']),
            new Middleware('permission:account-type.store', only: ['create', 'store']),
            new Middleware('permission:account-type.update', only: ['edit', 'update']),
            new Middleware('permission:account-type.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 1,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "name", "title" => "Type", "orderable" => true, "className" => "text-start"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "account-type.show"],
            ["type" => "edit", "url" => "account-type.edit"],
            ["type" => "delete", "url" => "account-type.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "name",
                "label" => "Type",
                "col" => 6
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = AccountType::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'account-type');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("account-type.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new AccountType();

        return view("account-type.form", [
            "model" => $model,
            "isBackUrl" => route("account-type.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AccountTypeRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->accountTypeRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('account-type.index')
                ->with('success', 'Account type created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create account type');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->accountTypeRepository->findOrFail($id);

        return view("account-type.show", [
            "model" => $model,
            "isBackUrl" => route("account-type.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->accountTypeRepository->findOrFail($id);

        return view("account-type.form", [
            "model" => $model,
            "isBackUrl" => route("account-type.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AccountTypeRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->accountTypeRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('account-type.index')
                ->with('success', 'Account type updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update account type');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->accountTypeRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Account type deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete account type');
        }
    }
}
