<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Activitylog\Models\Activity;
use App\Helpers\ActivityLogHelper;

class ActivityLogController extends Controller
{
    /**
     * Display a listing of activity logs.
     */
    public function index(Request $request)
    {
        $query = Activity::with(['causer', 'subject']);

        // Filter by model type - handle Transaction separation by type
        if ($request->filled('model_type')) {
            $modelType = $request->model_type;

            // Check if filtering for specific transaction types (quotation or salesorder)
            if ($modelType === 'quotation' || $modelType === 'salesorder') {
                $query->where('subject_type', 'App\\Models\\Transaction')
                      ->whereIn('subject_id', function($subquery) use ($modelType) {
                          $subquery->select('id')
                                   ->from('transactions')
                                   ->where('type', $modelType)
                                   ->whereNull('deleted_at');
                      });
            } else {
                $query->where('subject_type', $modelType);
            }
        }

        // Filter by event (created, updated, deleted)
        if ($request->filled('event')) {
            $query->where('event', $request->event);
        }

        // Filter by date range
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        // Filter by user
        if ($request->filled('user_id')) {
            $query->where('causer_id', $request->user_id);
        }

        // Search in description
        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        $activities = $query->latest()->paginate(50);

        // Get available model types for filter - separate Transaction by type
        $modelTypes = Activity::select('subject_type')
            ->distinct()
            ->orderBy('subject_type')
            ->pluck('subject_type')
            ->flatMap(function($type) {
                // Separate Transaction model by quotation and sales order
                if ($type === 'App\\Models\\Transaction') {
                    return [
                        ['value' => 'quotation', 'label' => 'Quotation'],
                        ['value' => 'salesorder', 'label' => 'Sales Order'],
                    ];
                }
                return [
                    ['value' => $type, 'label' => class_basename($type)]
                ];
            })
            ->sortBy('label')
            ->values();

        // Get all users for filter
        $users = \App\Models\User::select('id', 'name')
            ->orderBy('name')
            ->get();

        // Get available events
        $events = Activity::select('event')
            ->distinct()
            ->orderBy('event')
            ->pluck('event');

        return view('activity-log.index', compact('activities', 'modelTypes', 'users', 'events'));
    }

    /**
     * Display activity log for a specific model.
     */
    public function show($modelType, $modelId)
    {
        // Validate and get model class
        $modelClass = "App\\Models\\{$modelType}";

        if (!class_exists($modelClass)) {
            abort(404, 'Model not found');
        }

        $model = $modelClass::findOrFail($modelId);

        $activities = ActivityLogHelper::getActivitiesForModel($model);

        return view('activity-log.show', compact('activities', 'model'));
    }

    /**
     * Get activities for current user.
     */
    public function myActivities()
    {
        $activities = ActivityLogHelper::getActivitiesByCauser(auth()->id());

        return view('activity-log.my-activities', compact('activities'));
    }

    /**
     * Get activity detail (AJAX).
     */
    public function detail($activityId)
    {
        $activity = Activity::with(['causer', 'subject'])->findOrFail($activityId);

        $formatted = ActivityLogHelper::formatActivity($activity);
        $formatted['change_summary'] = ActivityLogHelper::getChangeSummary($activity);

        return response()->json($formatted);
    }

    /**
     * Export activity logs (CSV).
     */
    public function export(Request $request)
    {
        $query = Activity::with(['causer', 'subject']);

        // Apply same filters as index - handle Transaction separation by type
        if ($request->filled('model_type')) {
            $modelType = $request->model_type;

            // Check if filtering for specific transaction types (quotation or salesorder)
            if ($modelType === 'quotation' || $modelType === 'salesorder') {
                $query->where('subject_type', 'App\\Models\\Transaction')
                      ->whereIn('subject_id', function($subquery) use ($modelType) {
                          $subquery->select('id')
                                   ->from('transactions')
                                   ->where('type', $modelType)
                                   ->whereNull('deleted_at');
                      });
            } else {
                $query->where('subject_type', $modelType);
            }
        }

        if ($request->filled('event')) {
            $query->where('event', $request->event);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        if ($request->filled('user_id')) {
            $query->where('causer_id', $request->user_id);
        }

        $activities = $query->latest()->limit(10000)->get();

        $filename = 'activity-log-' . now()->format('Y-m-d-His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($activities) {
            $file = fopen('php://output', 'w');

            // Header row
            fputcsv($file, [
                'ID',
                'Date/Time',
                'User',
                'Action',
                'Model',
                'Model ID',
                'Description',
                'Changes'
            ]);

            // Data rows
            foreach ($activities as $activity) {
                fputcsv($file, [
                    $activity->id,
                    $activity->created_at->format('Y-m-d H:i:s'),
                    $activity->causer ? $activity->causer->name : 'System',
                    $activity->event,
                    class_basename($activity->subject_type),
                    $activity->subject_id,
                    $activity->description,
                    ActivityLogHelper::getChangeSummary($activity)
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
