<?php

namespace App\Http\Controllers;

use App\Http\Requests\ConfigRequest;
use App\Models\Config;
use App\Repositories\ConfigRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class ConfigController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $configRepository;

    public function __construct(ConfigRepository $configRepository)
    {
        $this->configRepository = $configRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:config.index', only: ['index']),
            new Middleware('permission:config.show', only: ['show']),
            new Middleware('permission:config.store', only: ['create', 'store']),
            new Middleware('permission:config.update', only: ['edit', 'update']),
            new Middleware('permission:config.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 2,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "key", "title" => "Key", "orderable" => true, 'className' => 'text-start'],
            ["data" => "value", "title" => "Value", "orderable" => true, 'className' => 'text-start'],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "config.show"],
            ["type" => "edit", "url" => "config.edit"],
            ["type" => "delete", "url" => "config.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "key",
                "label" => "Key",
                "col" => 6
            ],
            [
                "type" => "text",
                "name" => "value",
                "label" => "Value",
                "col" => 6
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Config::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'config');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("config.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Config();

        return view("config.form", [
            "model" => $model,
            "isBackUrl" => route("config.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ConfigRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            $this->configRepository->create($data);

            DB::commit();

            return redirect()
                ->route('config.index')
                ->with('success', 'Config created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create config');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->configRepository->findOrFail($id);

        return view("config.show", [
            "model" => $model,
            "isBackUrl" => route("config.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->configRepository->findOrFail($id);

        return view("config.form", [
            "model" => $model,
            "isBackUrl" => route("config.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ConfigRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            $this->configRepository->update($id, $data);

            DB::commit();

            return redirect()
                ->route('config.index')
                ->with('success', 'Config updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update config');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->configRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Config deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete config');
        }
    }
}
