<?php

namespace App\Http\Controllers;

use App\Http\Requests\CustomerRequest;
use App\Models\Customer;
use App\Repositories\CustomerRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class CustomerController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $customerRepository;

    public function __construct(CustomerRepository $customerRepository)
    {
        $this->customerRepository = $customerRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:customer.index', only: ['index']),
            new Middleware('permission:customer.show', only: ['show']),
            new Middleware('permission:customer.store', only: ['create', 'store']),
            new Middleware('permission:customer.update', only: ['edit', 'update']),
            new Middleware('permission:customer.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 3,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "full_name", "title" => "Name", "orderable" => true, 'className' => 'text-start'],
            ["data" => "phone", "title" => "Phone", "orderable" => true, 'className' => 'text-start'],
            ["data" => "address", "title" => "Address", "orderable" => true, 'className' => 'text-start'],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "customer.show"],
            ["type" => "edit", "url" => "customer.edit"],
            ["type" => "delete", "url" => "customer.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "full_name",
                "label" => "Name",
                "col" => 4
            ],
            [
                "type" => "text",
                "name" => "phone",
                "label" => "Phone",
                "col" => 4
            ],
            [
                "type" => "text",
                "name" => "address",
                "label" => "Address",
                "col" => 4
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Customer::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'customer');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("customer.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Customer();

        return view("customer.form", [
            "model" => $model,
            "isBackUrl" => route("customer.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CustomerRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            // Handle checkbox values
            $data['is_ppn'] = $request->has('is_ppn') ? true : false;
            $data['is_pph'] = $request->has('is_pph') ? true : false;

            $this->customerRepository->createCustomer($data);

            DB::commit();

            return redirect()
                ->route('customer.index')
                ->with('success', 'Customer created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create customer');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->customerRepository->findOrFail($id);
        $model->load(['quotations', 'salesOrders']);

        return view("customer.show", [
            "model" => $model,
            "isBackUrl" => route("customer.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->customerRepository->findOrFail($id);

        return view("customer.form", [
            "model" => $model,
            "isBackUrl" => route("customer.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(CustomerRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            // Handle checkbox values
            $data['is_ppn'] = $request->has('is_ppn') ? true : false;
            $data['is_pph'] = $request->has('is_pph') ? true : false;

            $this->customerRepository->update($id, $data);

            DB::commit();

            return redirect()
                ->route('customer.index')
                ->with('success', 'Customer updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update customer');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->customerRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Customer deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete customer');
        }
    }
}
