<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\DeliveryOrderRequest;
use App\Models\DeliveryOrder;
use App\Models\Driver;
use App\Models\Transaction;
use App\Models\Truck;
use App\Repositories\DeliveryOrderRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class DeliveryOrderController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $deliveryOrderRepository;

    public function __construct(DeliveryOrderRepository $deliveryOrderRepository)
    {
        $this->deliveryOrderRepository = $deliveryOrderRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:delivery-order.index', only: ['index']),
            new Middleware('permission:delivery-order.show', only: ['show']),
            new Middleware('permission:delivery-order.create', only: ['create', 'store']),
            new Middleware('permission:delivery-order.update', only: ['edit', 'update']),
            new Middleware('permission:delivery-order.destroy', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("delivery-order.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 4,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "code", "title" => "DO Code", "orderable" => true, 'className' => 'text-start'],
            ["data" => "id_transaction", "title" => "Sales Order", "orderable" => true, 'className' => 'text-start'],
            ["data" => "customer_name", "title" => "Customer", "orderable" => false, "className" => "text-start", "exclude_column" => true],
            ["data" => "total_routes", "title" => "Total Routes", "orderable" => false, 'className' => 'text-center', "exclude_column" => true],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-center'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getFormatters()
    {
        return [
            'id_transaction' => function ($row, $value) {
                if (!$row->transaction) {
                    return '-';
                }

                $code = $row->transaction->code;

                if (auth()->user()->can('sales-order.show')) {
                    return "<a href='" . route('sales-order.show', $row->transaction->id) . "'>$code</a>";
                }

                return $code;
            },
            'customer_name' => function ($row, $value) {
                if (!$row->transaction || !$row->transaction->customer) {
                    return '-';
                }

                $fullName = $row->transaction->customer->full_name;

                if (auth()->user()->can('customer.show')) {
                    return "<a href='" . route('customer.show', $row->transaction->customer->id) . "'>$fullName</a>";
                }

                return $fullName;
            },
            'total_routes' => function ($row, $value) {
                return $row->details ? $row->details->count() : 0;
            },
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "delivery-order.show"],
            ["type" => "edit", "url" => "delivery-order.edit"],
            ["type" => "delete", "url" => "delivery-order.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "code",
                "label" => "DO Code",
                "col" => 4
            ],
            [
                "type" => "date",
                "name" => "created_at",
                "label" => "Created Date",
                "col" => 4
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = DeliveryOrder::with(['transaction.customer', 'details']);

        $actions = $this->addPermissionsToActions($this->getActions(), 'delivery-order');
        $result = $this->processDataTable(
            $request,
            $query,
            $this->getColumns(),
            $actions,
            $this->getFilters(),
            $this->getFormatters()
        );

        return response()->json($result);
    }

    /**
     * Show the form for creating a new delivery order from sales order.
     */
    public function create(string $sales_order)
    {
        $salesOrder = Transaction::with(['customer', 'sales', 'details.origin', 'details.destination', 'details.loadout'])
            ->where('type', 'salesorder')
            ->findOrFail($sales_order);

        // Check if sales order can create delivery order
        if ($salesOrder->status !== 'pending') {
            return redirect()
                ->route('sales-order.show', $salesOrder->id)
                ->with('error', 'Only pending sales orders can create delivery order.');
        }

        // Check if delivery order already exists for this sales order
        $existingDeliveryOrder = DeliveryOrder::where('id_transaction', $salesOrder->id)->first();

        if ($existingDeliveryOrder) {
            return redirect()
                ->route('delivery-order.show', $existingDeliveryOrder->id)
                ->with('info', 'Delivery order already exists for this sales order.');
        }

        $model = new DeliveryOrder();
        $drivers = Driver::orderBy('full_name')->get();
        $trucks = Truck::with('type')->orderBy('license_plate')->get();

        return view("delivery-order.form", [
            "model" => $model,
            "salesOrder" => $salesOrder,
            "drivers" => $drivers,
            "trucks" => $trucks,
            "isBackUrl" => route("sales-order.show", $salesOrder->id)
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(DeliveryOrderRequest $request)
    {
        try {
            $validated = $request->validated();

            // Get sales order
            $salesOrderId = $request->input('id_transaction');
            $salesOrder = Transaction::where('type', 'salesorder')->findOrFail($salesOrderId);

            // Check if sales order can create delivery order
            if ($salesOrder->status !== 'pending') {
                return redirect()
                    ->route('sales-order.show', $salesOrderId)
                    ->with('error', 'Only pending sales orders can create delivery order.');
            }

            // Create delivery order
            $deliveryOrder = $this->deliveryOrderRepository->createDeliveryOrder($validated, $salesOrder);

            return redirect()
                ->route('delivery-order.show', $deliveryOrder->id)
                ->with('success', 'Delivery order created successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to create delivery order');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->deliveryOrderRepository->findOrFail((int) $id);
        $model->load(['transaction.customer', 'transaction.sales', 'details.detail.origin', 'details.detail.destination', 'details.detail.loadout', 'details.driver', 'details.truck.type', 'createdBy', 'updatedBy']);

        return view("delivery-order.show", [
            "model" => $model,
            "isBackUrl" => route("delivery-order.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->deliveryOrderRepository->findOrFail((int) $id);
        $model->load(['transaction.details.origin', 'transaction.details.destination', 'transaction.details.loadout', 'details.driver', 'details.truck']);

        $drivers = Driver::orderBy('full_name')->get();
        $trucks = Truck::with('type')->orderBy('license_plate')->get();

        return view("delivery-order.form", [
            "model" => $model,
            "salesOrder" => $model->transaction,
            "drivers" => $drivers,
            "trucks" => $trucks,
            "isBackUrl" => route("delivery-order.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(DeliveryOrderRequest $request, string $id)
    {
        try {
            $validated = $request->validated();

            // Update delivery order
            $deliveryOrder = $this->deliveryOrderRepository->updateDeliveryOrder((int) $id, $validated);

            return redirect()
                ->route('delivery-order.show', $deliveryOrder->id)
                ->with('success', 'Delivery order updated successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to update delivery order');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->deliveryOrderRepository->softDelete((int) $id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Delivery order deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete delivery order');
        }
    }
}
