<?php

namespace App\Http\Controllers;

use App\Http\Requests\DriverRequest;
use App\Models\Driver;
use App\Repositories\DriverRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class DriverController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $driverRepository;

    public function __construct(DriverRepository $driverRepository)
    {
        $this->driverRepository = $driverRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:driver.index', only: ['index']),
            new Middleware('permission:driver.show', only: ['show']),
            new Middleware('permission:driver.store', only: ['create', 'store']),
            new Middleware('permission:driver.update', only: ['edit', 'update']),
            new Middleware('permission:driver.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 5,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "code", "title" => "Code", "orderable" => true, "className" => "text-start"],
            ["data" => "full_name", "title" => "Full Name", "orderable" => true, "className" => "text-start"],
            ["data" => "phone", "title" => "Phone", "orderable" => true, "className" => "text-start"],
            ["data" => "license_number", "title" => "License Number", "orderable" => true, "className" => "text-start"],
            ["data" => "status", "title" => "Status", "orderable" => true, "className" => "text-center"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getFormatters()
    {
        return [
            'status' => function ($row, $value) {

                $statusColor = $row->getStatusColor();
                $status = strtoupper($row->status);

                return "<span class='badge badge-sm bg-$statusColor'>$status</span>";
            }
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "driver.show"],
            ["type" => "edit", "url" => "driver.edit"],
            ["type" => "delete", "url" => "driver.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "code",
                "label" => "Code",
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "full_name",
                "label" => "Full Name",
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "phone",
                "label" => "Phone",
                "col" => 3
            ],
            [
                "type" => "select",
                "name" => "status",
                "label" => "Status",
                "options" => [
                    "active" => "Active",
                    "blacklist" => "Blacklist",
                    "resigned" => "Resigned",
                ],
                "col" => 3
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Driver::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'driver');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters(), $this->getFormatters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("driver.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Driver();

        return view("driver.form", [
            "model" => $model,
            "isBackUrl" => route("driver.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(DriverRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            // Handle file uploads
            if ($request->hasFile('identity_file')) {
                $data['identity_file'] = $request->file('identity_file')->store('drivers/identity', 'public');
            }
            if ($request->hasFile('license_file')) {
                $data['license_file'] = $request->file('license_file')->store('drivers/license', 'public');
            }
            if ($request->hasFile('kk_file')) {
                $data['kk_file'] = $request->file('kk_file')->store('drivers/kk', 'public');
            }
            if ($request->hasFile('photo_file')) {
                $data['photo_file'] = $request->file('photo_file')->store('drivers/photo', 'public');
            }
            if ($request->hasFile('skck_file')) {
                $data['skck_file'] = $request->file('skck_file')->store('drivers/skck', 'public');
            }

            $this->driverRepository->create($data);

            DB::commit();

            return redirect()
                ->route('driver.index')
                ->with('success', 'Driver created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create driver');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->driverRepository->findOrFail($id);

        return view("driver.show", [
            "model" => $model,
            "isBackUrl" => route("driver.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->driverRepository->findOrFail($id);

        return view("driver.form", [
            "model" => $model,
            "isBackUrl" => route("driver.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(DriverRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $driver = $this->driverRepository->findOrFail($id);
            $data = $request->validated();

            // Handle file uploads and delete old files
            if ($request->hasFile('identity_file')) {
                if ($driver->identity_file && Storage::disk('public')->exists($driver->identity_file)) {
                    Storage::disk('public')->delete($driver->identity_file);
                }
                $data['identity_file'] = $request->file('identity_file')->store('drivers/identity', 'public');
            }
            if ($request->hasFile('license_file')) {
                if ($driver->license_file && Storage::disk('public')->exists($driver->license_file)) {
                    Storage::disk('public')->delete($driver->license_file);
                }
                $data['license_file'] = $request->file('license_file')->store('drivers/license', 'public');
            }
            if ($request->hasFile('kk_file')) {
                if ($driver->kk_file && Storage::disk('public')->exists($driver->kk_file)) {
                    Storage::disk('public')->delete($driver->kk_file);
                }
                $data['kk_file'] = $request->file('kk_file')->store('drivers/kk', 'public');
            }
            if ($request->hasFile('photo_file')) {
                if ($driver->photo_file && Storage::disk('public')->exists($driver->photo_file)) {
                    Storage::disk('public')->delete($driver->photo_file);
                }
                $data['photo_file'] = $request->file('photo_file')->store('drivers/photo', 'public');
            }
            if ($request->hasFile('skck_file')) {
                if ($driver->skck_file && Storage::disk('public')->exists($driver->skck_file)) {
                    Storage::disk('public')->delete($driver->skck_file);
                }
                $data['skck_file'] = $request->file('skck_file')->store('drivers/skck', 'public');
            }

            $this->driverRepository->update($id, $data);

            DB::commit();

            return redirect()
                ->route('driver.index')
                ->with('success', 'Driver updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update driver');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->driverRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Driver deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete driver');
        }
    }
}
