<?php

namespace App\Http\Controllers;

use App\Models\Journal;
use App\Models\Account;
use App\Models\AccountJournal;
use App\Http\Requests\StoreJournalRequest;
use App\Http\Requests\UpdateJournalRequest;
use App\Repositories\JournalRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class JournalController extends Controller
{
    use DataTableTrait, HandlesExceptions;

    protected $journalRepository;

    public function __construct(JournalRepository $journalRepository)
    {
        $this->journalRepository = $journalRepository;
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 3,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "code", "title" => "Code", "orderable" => true, "className" => "text-start"],
            ["data" => "date", "title" => "Date", "orderable" => true, "className" => "text-start"],
            ["data" => "description", "title" => "Description", "orderable" => true, "className" => "text-start"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "journal.show"],
            ["type" => "edit", "url" => "journal.edit"],
            ["type" => "delete", "url" => "journal.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "code",
                "label" => "Code",
                "col" => 4
            ],
            [
                "type" => "date",
                "name" => "date",
                "label" => "Date",
                "col" => 4
            ],
            [
                "type" => "text",
                "name" => "description",
                "label" => "Description",
                "col" => 4
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Journal::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'journal');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("journal.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Journal();
        $accounts = Account::orderBy('code')->get();

        return view("journal.form", [
            "model" => $model,
            "accounts" => $accounts,
            "isBackUrl" => route("journal.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreJournalRequest $request)
    {
        try {
            $this->journalRepository->createJournal($request->validated());

            return redirect()->route('journal.index')->with('success', 'Journal created successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to create journal');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = Journal::with('accountJournals.account')->findOrFail($id);

        return view("journal.show", [
            "model" => $model,
            "isBackUrl" => route("journal.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = Journal::with('accountJournals.account')->findOrFail($id);
        $accounts = Account::orderBy('code')->get();

        return view("journal.form", [
            "model" => $model,
            "accounts" => $accounts,
            "isBackUrl" => route("journal.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateJournalRequest $request, string $id)
    {
        try {
            $this->journalRepository->updateJournal($id, $request->validated());

            return redirect()->route('journal.index')->with('success', 'Journal updated successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to update journal');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $this->journalRepository->deleteJournal($id);

            return redirect()->route('journal.index')->with('success', 'Journal deleted successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to delete journal');
        }
    }
}
