<?php

namespace App\Http\Controllers;

use App\Http\Requests\LocationRequest;
use App\Models\Location;
use App\Repositories\LocationRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class LocationController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $locationRepository;

    public function __construct(LocationRepository $locationRepository)
    {
        $this->locationRepository = $locationRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:location.index', only: ['index']),
            new Middleware('permission:location.show', only: ['show']),
            new Middleware('permission:location.store', only: ['create', 'store']),
            new Middleware('permission:location.update', only: ['edit', 'update']),
            new Middleware('permission:location.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 1,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "name", "title" => "Location Name", "orderable" => true, 'className' => 'text-start'],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "location.show"],
            ["type" => "edit", "url" => "location.edit"],
            ["type" => "delete", "url" => "location.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "name",
                "label" => "Location Name",
                "col" => 12
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Location::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'location');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("location.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Location();

        return view("location.form", [
            "model" => $model,
            "isBackUrl" => route("location.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(LocationRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->locationRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('location.index')
                ->with('success', 'Location created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create location');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->locationRepository->findOrFail($id);

        return view("location.show", [
            "model" => $model,
            "isBackUrl" => route("location.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->locationRepository->findOrFail($id);

        return view("location.form", [
            "model" => $model,
            "isBackUrl" => route("location.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(LocationRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->locationRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('location.index')
                ->with('success', 'Location updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update location');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->locationRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Location deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete location');
        }
    }
}
