<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\PriceRequest;
use App\Models\Price;
use App\Models\Route;
use App\Models\TruckLoadout;
use App\Models\Customer;
use App\Repositories\PriceRepository;
use App\Repositories\RouteRepository;
use App\Repositories\TruckLoadoutRepository;
use App\Repositories\CustomerRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class PriceController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $priceRepository;
    protected $routeRepository;
    protected $truckLoadoutRepository;
    protected $customerRepository;

    public function __construct(
        PriceRepository $priceRepository,
        RouteRepository $routeRepository,
        TruckLoadoutRepository $truckLoadoutRepository,
        CustomerRepository $customerRepository
    ) {
        $this->priceRepository = $priceRepository;
        $this->routeRepository = $routeRepository;
        $this->truckLoadoutRepository = $truckLoadoutRepository;
        $this->customerRepository = $customerRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:price.index', only: ['index']),
            new Middleware('permission:price.show', only: ['show']),
            new Middleware('permission:price.store', only: ['create', 'store']),
            new Middleware('permission:price.update', only: ['edit', 'update']),
            new Middleware('permission:price.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 4,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "id_route", "title" => "Route", "orderable" => true, "className" => "text-start"],
            ["data" => "id_loadout", "title" => "Loadout", "orderable" => true, "className" => "text-start"],
            ["data" => "id_customer", "title" => "Customer", "orderable" => true, "className" => "text-start"],
            ["data" => "price", "title" => "Price", "orderable" => true, "className" => "text-end"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getFormatters()
    {
        return [
            'id_route' => function ($row, $value) {
                if ($row->route && $row->route->origin && $row->route->destination) {
                    return $row->route->origin->name . ' - ' . $row->route->destination->name;
                }
                return $value;
            },
            'id_customer' => function ($row, $value) {
                return $row->customer ? $row->customer->full_name : null;
            },
            'id_loadout' => function ($row, $value) {
                return $row->loadout ? $row->loadout->name : null;
            },
            'price' => function ($row, $value) {
                return Helpers::formatCurrency($value, 2);
            }
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "price.show"],
            ["type" => "edit", "url" => "price.edit"],
            ["type" => "delete", "url" => "price.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "select",
                "name" => "id_route",
                "label" => "Route",
                "options" => Helpers::toDropdownOptions(Route::all(), "name", "id"),
                "col" => 3,
                "searchable" => true
            ],
            [
                "type" => "select",
                "name" => "id_loadout",
                "label" => "Loadout",
                "options" => Helpers::toDropdownOptions(TruckLoadout::all(), "name", "id"),
                "col" => 3
            ],
            [
                "type" => "select",
                "name" => "id_customer",
                "label" => "Customer",
                "options" => Helpers::toDropdownOptions(Customer::all(), "full_name", "id"),
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "price",
                "label" => "Price",
                "col" => 3
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Price::query()->with(['route.origin', 'route.destination']);

        $actions = $this->addPermissionsToActions($this->getActions(), 'price');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters(), $this->getFormatters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("price.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Price();
        $routes = Route::with(['origin', 'destination'])->get();
        $loadouts = $this->truckLoadoutRepository->all();
        $customers = $this->customerRepository->all();

        return view("price.form", [
            "model" => $model,
            "routes" => $routes,
            "loadouts" => $loadouts,
            "customers" => $customers,
            "isBackUrl" => route("price.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(PriceRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->priceRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('price.index')
                ->with('success', 'Price created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create price');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = Price::with(['route.origin', 'route.destination', 'loadout', 'customer'])->findOrFail($id);

        return view("price.show", [
            "model" => $model,
            "isBackUrl" => route("price.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->priceRepository->findOrFail($id);
        $routes = Route::with(['origin', 'destination'])->get();
        $loadouts = $this->truckLoadoutRepository->all();
        $customers = $this->customerRepository->all();

        return view("price.form", [
            "model" => $model,
            "routes" => $routes,
            "loadouts" => $loadouts,
            "customers" => $customers,
            "isBackUrl" => route("price.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(PriceRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->priceRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('price.index')
                ->with('success', 'Price updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update price');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->priceRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Price deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete price');
        }
    }
}
