<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\QuotationRequest;
use App\Models\Customer;
use App\Models\Transaction;
use App\Models\User;
use App\Repositories\QuotationRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class QuotationController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $quotationRepository;

    public function __construct(QuotationRepository $quotationRepository)
    {
        $this->quotationRepository = $quotationRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:quotation.index', only: ['index']),
            new Middleware('permission:quotation.show', only: ['show']),
            new Middleware('permission:quotation.store', only: ['create', 'store']),
            new Middleware('permission:quotation.update', only: ['edit', 'update']),
            new Middleware('permission:quotation.destroy', only: ['destroy']),
            new Middleware('permission:quotation.reject', only: ['reject']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("quotation.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 6,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "code", "title" => "Code", "orderable" => true, 'className' => 'text-start'],
            ["data" => "id_customer", "title" => "Customer", "orderable" => true, "className" => "text-start"],
            ["data" => "id_sales", "title" => "Sales", "orderable" => true, "className" => "text-start"],
            ["data" => "invoice_date", "title" => "Invoice Date", "orderable" => true, 'className' => 'text-center'],
            ["data" => "grand_total", "title" => "Grand Total", "orderable" => true, 'className' => 'text-end'],
            ["data" => "status", "title" => "Status", "orderable" => true, 'className' => 'text-center'],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-center'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getFormatters()
    {
        return [
            'id_customer' => function ($row, $value) {
                if (!$row->customer) {
                    return null;
                }

                $fullName = $row->customer->full_name;

                if (auth()->user()->can('customer.show')) {
                    return "<a href='" . route('customer.show', $row->customer->id) . "'>$fullName</a>";
                }

                return $fullName;
            },
            'id_sales' => function ($row, $value) {
                return $row->sales ? $row->sales->name : null;
            },
            'status' => function ($row, $value) {

                $statusColor = $row->getStatusColor();
                $status = strtoupper($row->status);

                return "<span class='badge badge-sm bg-$statusColor'>$status</span>";
            },
            'grand_total' => function ($row, $value) {
                return Helpers::formatCurrency($value, 2);
            }
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "quotation.show"],
            ["type" => "edit", "url" => "quotation.edit", "condition" => ["status" => ["pending"]]],
            ["type" => "delete", "url" => "quotation.destroy", "condition" => ["status" => ["pending"]]],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "code",
                "label" => "Code",
                "col" => 4
            ],
            [
                "type" => "select",
                "name" => "id_customer",
                "label" => "Customer",
                "options" => Customer::orderBy('full_name')->pluck('full_name', 'id')->toArray(),
                "col" => 4
            ],
            [
                "type" => "select",
                "name" => "id_sales",
                "label" => "Sales",
                "options" => User::role('SALES')->orderBy('name')->pluck('name', 'id')->toArray(),
                "col" => 4
            ],
            [
                "type" => "select",
                "name" => "status",
                "label" => "Status",
                "options" => [
                    "pending" => "Pending",
                    "approved" => "Approved",
                    "rejected" => "Rejected",
                    "cancel" => "Cancel"
                ],
                "col" => 4
            ],
            [
                "type" => "date",
                "name" => "invoice_date",
                "label" => "Invoice Date",
                "col" => 4
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Transaction::with(['customer', 'sales'])
            ->where('type', 'quotation')->withTrashed();

        $actions = $this->addPermissionsToActions($this->getActions(), 'quotation');
        $result = $this->processDataTable(
            $request,
            $query,
            $this->getColumns(),
            $actions,
            $this->getFilters(),
            $this->getFormatters()
        );

        return response()->json($result);
    }

    /**
     * Show the form for creating a new quotation.
     */
    public function create()
    {
        $model = new Transaction();
        $formData = $this->quotationRepository->getFormData();

        return view("quotation.form", array_merge(
            ['model' => $model],
            $formData,
            ['isBackUrl' => route("dashboard")]
        ));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(QuotationRequest $request)
    {
        try {
            $validated = $request->validated();

            // Prepare quotation data with calculations
            $preparedData = $this->quotationRepository->prepareQuotationData($validated);

            // Create quotation
            $quotation = $this->quotationRepository->createQuotation($preparedData);

            return redirect()
                ->route('quotation.show', $quotation->id)
                ->with('success', 'Quotation created successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to create quotation');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->quotationRepository->findOrFail($id, true);
        $model->load(['approvedBy', 'rejectedBy', 'createdBy', 'updatedBy']);

        return view("quotation.show", [
            "model" => $model,
            "isBackUrl" => route("quotation.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->quotationRepository->findOrFail($id);
        $model->load('details.origin', 'details.destination', 'details.loadout');

        // Only allow editing pending quotations
        if ($model->status !== 'pending') {
            return redirect()
                ->route('quotation.index')
                ->with('error', 'Only pending quotations can be edited.');
        }

        $formData = $this->quotationRepository->getFormData();

        return view("quotation.form", array_merge(
            ['model' => $model],
            $formData,
            ['isBackUrl' => route("quotation.index")]
        ));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(QuotationRequest $request, string $id)
    {
        try {
            $model = $this->quotationRepository->findOrFail($id);

            // Only allow updating pending quotations
            if ($model->status !== 'pending') {
                return redirect()
                    ->route('quotation.index')
                    ->with('error', 'Only pending quotations can be updated.');
            }

            $validated = $request->validated();

            // Prepare quotation data with calculations
            $preparedData = $this->quotationRepository->prepareQuotationData($validated);

            // Update quotation
            $quotation = $this->quotationRepository->updateQuotation($id, $preparedData);

            return redirect()
                ->route('quotation.show', $quotation->id)
                ->with('success', 'Quotation updated successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to update quotation');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $model = $this->quotationRepository->findOrFail($id);

            // Only allow deleting pending quotations
            if ($model->status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only pending quotations can be deleted.'
                ], 400);
            }

            $this->quotationRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Quotation deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete quotation');
        }
    }

    /**
     * Get prices for a specific route and customer
     */
    public function getPrices(Request $request)
    {
        $routeId = $request->input('route_id');
        $customerId = $request->input('customer_id');

        $prices = $this->quotationRepository->getPricesForRoute($routeId, $customerId);

        return response()->json([
            'success' => true,
            'data' => $prices
        ]);
    }

    public function getRouteAllowances(Request $request)
    {
        $routeId = $request->input('route_id');

        $allowances = $this->quotationRepository->getRouteAllowances($routeId);

        if (!$allowances) {
            return response()->json([
                'success' => false,
                'message' => 'Route not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $allowances
        ]);
    }

    /**
     * Reject the quotation.
     */
    public function reject(string $id)
    {
        try {
            $model = $this->quotationRepository->findOrFail($id);

            // Only allow rejecting pending quotations
            if ($model->status !== 'pending') {
                return redirect()
                    ->route('quotation.show', $id)
                    ->with('error', 'Only pending quotations can be rejected.');
            }

            $this->quotationRepository->rejectQuotation($id);

            return redirect()
                ->route('quotation.show', $id)
                ->with('success', 'Quotation has been rejected.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to reject quotation');
        }
    }
}
