<?php

namespace App\Http\Controllers;

use App\Http\Requests\RoleRequest;
use App\Models\Role;
use App\Repositories\RoleRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class RoleController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $roleRepository;

    public function __construct(RoleRepository $roleRepository)
    {
        $this->roleRepository = $roleRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:role.index', only: ['index']),
            new Middleware('permission:role.show', only: ['show']),
            new Middleware('permission:role.store', only: ['create', 'store']),
            new Middleware('permission:role.update', only: ['edit', 'update']),
            new Middleware('permission:role.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 1,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "name", "title" => "Role Name", "orderable" => true, 'className' => 'text-start'],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "role.show"],
            ["type" => "edit", "url" => "role.edit"],
            ["type" => "delete", "url" => "role.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "name",
                "label" => "Role Name",
                "col" => 4
            ],
        ];
    }

    private function getAllPermissions()
    {
        // Get resources and exclusions from config file
        $resources = config('permissions.resources', []);
        $exclusions = config('permissions.excluded_permissions', []);

        // Define standard CRUD actions with their labels and actual permissions they control
        $standardActions = [
            'index' => [
                'label' => 'View All',
                'permissions' => ['index']
            ],
            'show' => [
                'label' => 'View Detail',
                'permissions' => ['show']
            ],
            'create' => [
                'label' => 'Create',
                'permissions' => ['create', 'store']
            ],
            'update' => [
                'label' => 'Update',
                'permissions' => ['edit', 'update']
            ],
            'destroy' => [
                'label' => 'Delete',
                'permissions' => ['destroy']
            ],
        ];

        // Build permissions array
        $permissions = [];
        foreach ($resources as $resourceKey => $resourceLabel) {
            // Skip if entire resource is excluded
            if ($this->isExcluded($resourceKey, '*', $exclusions)) {
                continue;
            }

            $resourcePermissions = [];
            foreach ($standardActions as $actionKey => $actionData) {
                // Check if any of the actual permissions are excluded
                $shouldInclude = true;
                foreach ($actionData['permissions'] as $perm) {
                    if ($this->isExcluded($resourceKey, $perm, $exclusions)) {
                        $shouldInclude = false;
                        break;
                    }
                }

                if ($shouldInclude) {
                    $resourcePermissions[] = [
                        'key' => $actionKey,
                        'label' => $actionData['label'] . ($actionKey === 'index' || $actionKey === 'create' || $actionKey === 'update' ? ' ' . $resourceLabel : ''),
                        'resource_key' => $resourceKey,
                        'actual_permissions' => array_map(fn($p) => "{$resourceKey}.{$p}", $actionData['permissions'])
                    ];
                }
            }

            // Only add resource if it has at least one permission
            if (!empty($resourcePermissions)) {
                $permissions[$resourceLabel] = $resourcePermissions;
            }
        }

        return $permissions;
    }

    /**
     * Check if a permission is excluded
     */
    private function isExcluded($resource, $action, $exclusions)
    {
        foreach ($exclusions as $pattern) {
            // Check exact match: resource.action
            if ($pattern === "{$resource}.{$action}") {
                return true;
            }

            // Check wildcard resource: *.action
            if (str_starts_with($pattern, '*.') && str_ends_with($pattern, ".{$action}")) {
                return true;
            }

            // Check wildcard action: resource.*
            if (str_starts_with($pattern, "{$resource}.") && str_ends_with($pattern, '.*')) {
                return true;
            }

            // Check complete wildcard for resource
            if ($pattern === "{$resource}.*" && $action === '*') {
                return true;
            }
        }

        return false;
    }

    /**
     * Get all custom permissions from config file
     */
    private function getAllCustomPermissions()
    {
        $customConfig = config('permissions.custom_permissions', []);
        $exclusions = config('permissions.excluded_permissions', []);
        $customPermissions = [];

        foreach ($customConfig as $resource => $actions) {
            foreach ($actions as $action => $label) {
                // Skip if this custom permission is excluded
                if ($this->isExcluded($resource, $action, $exclusions)) {
                    continue;
                }

                if (!isset($customPermissions[$resource])) {
                    $customPermissions[$resource] = [];
                }
                $customPermissions[$resource][] = [
                    'name' => "{$resource}.{$action}",
                    'action' => $action,
                    'label' => $label
                ];
            }
        }

        return $customPermissions;
    }

    public function getData(Request $request)
    {
        $query = $this->roleRepository->queryWithPermissionCount();

        $actions = $this->addPermissionsToActions($this->getActions(), 'role');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("role.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Role();
        $permissions = $this->getAllPermissions();
        $customPermissions = $this->getAllCustomPermissions();

        return view("role.form", [
            "model" => $model,
            "permissions" => $permissions,
            "customPermissions" => $customPermissions,
            "isBackUrl" => route("role.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(RoleRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = [
                'name' => $request->name,
                'guard_name' => 'web',
            ];

            $role = $this->roleRepository->create($data);

            // Sync permissions - flatten the actual permissions array
            if ($request->has('permissions')) {
                $actualPermissions = $this->flattenPermissions($request->permissions);
                $this->roleRepository->syncPermissions($role, $actualPermissions);
            }

            DB::commit();

            return redirect()
                ->route('role.index')
                ->with('success', 'Role created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create role');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->roleRepository->findWithPermissions($id);

        return view("role.show", [
            "model" => $model,
            "isBackUrl" => route("role.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->roleRepository->findWithPermissions($id);
        $permissions = $this->getAllPermissions();
        $customPermissions = $this->getAllCustomPermissions();

        return view("role.form", [
            "model" => $model,
            "permissions" => $permissions,
            "customPermissions" => $customPermissions,
            "isBackUrl" => route("role.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(RoleRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $role = $this->roleRepository->findOrFail($id);

            $this->roleRepository->update($id, [
                'name' => $request->name,
            ]);

            // Sync permissions - flatten the actual permissions array
            $actualPermissions = [];
            if ($request->has('permissions')) {
                $actualPermissions = $this->flattenPermissions($request->permissions);
            }
            $this->roleRepository->syncPermissions($role, $actualPermissions);

            DB::commit();

            return redirect()
                ->route('role.index')
                ->with('success', 'Role updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update role');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->roleRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Role deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete role');
        }
    }

    /**
     * Flatten permissions array from the form submission
     */
    private function flattenPermissions(array $permissions): array
    {
        $actualPermissions = [];
        foreach ($permissions as $permissionGroup) {
            if (is_array($permissionGroup)) {
                $actualPermissions = array_merge($actualPermissions, $permissionGroup);
            }
        }
        return array_unique($actualPermissions);
    }
}
