<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\RouteRequest;
use App\Models\Route;
use App\Models\Location;
use App\Repositories\RouteRepository;
use App\Repositories\LocationRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Symfony\Component\Console\Helper\Helper;

class RouteController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $routeRepository;
    protected $locationRepository;

    public function __construct(RouteRepository $routeRepository, LocationRepository $locationRepository)
    {
        $this->routeRepository = $routeRepository;
        $this->locationRepository = $locationRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:route.index', only: ['index']),
            new Middleware('permission:route.show', only: ['show']),
            new Middleware('permission:route.store', only: ['create', 'store']),
            new Middleware('permission:route.update', only: ['edit', 'update']),
            new Middleware('permission:route.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 4,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "id_origin", "title" => "Origin", "orderable" => false, 'className' => 'text-start'],
            ["data" => "id_destination", "title" => "Destination", "orderable" => false, 'className' => 'text-start'],
            ["data" => "distance", "title" => "Distance (km)", "orderable" => true, 'className' => 'text-end'],
            ["data" => "duration", "title" => "Duration (hrs)", "orderable" => true, 'className' => 'text-end'],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getFormatters()
    {
        return [
            'distance' => function ($row, $value) {
                return Helpers::formatNumber($value, 2);
            },
            'duration' => function ($row, $value) {
                return Helpers::formatNumber($value, 2);
            }
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "route.show"],
            ["type" => "edit", "url" => "route.edit"],
            ["type" => "delete", "url" => "route.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "select",
                "name" => "id_origin",
                "label" => "Origin",
                "col" => 4,
                "options" => Location::orderBy('name')->get()->pluck('name', 'id')->toArray()
            ],
            [
                "type" => "select",
                "name" => "id_destination",
                "label" => "Destination",
                "col" => 4,
                "options" => Location::orderBy('name')->get()->pluck('name', 'id')->toArray()
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Route::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'route');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters(), $this->getFormatters());

        // Transform data for display - replace id_origin and id_destination with location names
        $result['data'] = collect($result['data'])->map(function ($item) {
            $route = Route::with(['origin', 'destination'])->find($item['id']);
            if ($route) {
                $item['id_origin'] = $route->origin->name ?? '-';
                $item['id_destination'] = $route->destination->name ?? '-';
            }
            return $item;
        })->toArray();

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("route.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Route();
        $locations = $this->locationRepository->all();

        return view("route.form", [
            "model" => $model,
            "locations" => $locations,
            "isBackUrl" => route("route.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(RouteRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->routeRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('route.index')
                ->with('success', 'Route created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create route');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->routeRepository->findOrFail($id);

        return view("route.show", [
            "model" => $model,
            "isBackUrl" => route("route.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->routeRepository->findOrFail($id);
        $locations = $this->locationRepository->all();

        return view("route.form", [
            "model" => $model,
            "locations" => $locations,
            "isBackUrl" => route("route.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(RouteRequest $request, string $id)
    {
        DB::beginTransaction();

        try {   
            $this->routeRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('route.index')
                ->with('success', 'Route updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update route');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->routeRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Route deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete route');
        }
    }
}
