<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\SalesOrderRequest;
use App\Models\Config;
use App\Models\Customer;
use App\Models\Location;
use App\Models\Price;
use App\Models\Role;
use App\Models\Route;
use App\Models\Transaction;
use App\Models\TruckLoadout;
use App\Models\User;
use App\Repositories\SalesOrderRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class SalesOrderController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $salesOrderRepository;

    public function __construct(SalesOrderRepository $salesOrderRepository)
    {
        $this->salesOrderRepository = $salesOrderRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:sales-order.index', only: ['index']),
            new Middleware('permission:sales-order.show', only: ['show']),
            new Middleware('permission:quotation.approve', only: ['create', 'store']),
            new Middleware('permission:sales-order.update', only: ['edit', 'update']),
            new Middleware('permission:sales-order.destroy', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("sales-order.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 2,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "code", "title" => "Code", "orderable" => true, 'className' => 'text-start'],
            ["data" => "id_transaction", "title" => "Quotation", "orderable" => true, 'className' => 'text-start'],
            ["data" => "id_customer", "title" => "Customer", "orderable" => true, "className" => "text-start"],
            ["data" => "id_sales", "title" => "Sales", "orderable" => true, "className" => "text-start"],
            ["data" => "invoice_date", "title" => "Invoice Date", "orderable" => true, 'className' => 'text-center'],
            ["data" => "grand_total", "title" => "Grand Total", "orderable" => true, 'className' => 'text-end'],
            ["data" => "status", "title" => "Status", "orderable" => true, 'className' => 'text-center'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getFormatters()
    {
        return [
            'id_transaction' => function ($row, $value) {
                if (!$row->parentTransaction) {
                    return '-';
                }

                $code = $row->parentTransaction->code;

                if (auth()->user()->can('quotation.show')) {
                    return "<a href='" . route('quotation.show', $row->parentTransaction->id) . "'>$code</a>";
                }

                return $code;
            },
            'id_customer' => function ($row, $value) {
                if (!$row->customer) {
                    return null;
                }

                $fullName = $row->customer->full_name;

                if (auth()->user()->can('customer.show')) {
                    return "<a href='" . route('customer.show', $row->customer->id) . "'>$fullName</a>";
                }

                return $fullName;
            },
            'id_sales' => function ($row, $value) {
                return $row->sales ? $row->sales->name : null;
            },
            'status' => function ($row, $value) {
                $statusColor = $row->getStatusColor();
                $status = strtoupper($row->status);

                return "<span class='badge badge-sm bg-$statusColor'>$status</span>";
            },
            'grand_total' => function ($row, $value) {
                return Helpers::formatCurrency($value, 2);
            }
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "sales-order.show"],
            ["type" => "edit", "url" => "sales-order.edit", "condition" => ["status" => ["pending"]]],
            ["type" => "delete", "url" => "sales-order.destroy", "condition" => ["status" => ["pending"]]],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "code",
                "label" => "Code",
                "col" => 4
            ],
            [
                "type" => "select",
                "name" => "id_customer",
                "label" => "Customer",
                "options" => Customer::orderBy('full_name')->pluck('full_name', 'id')->toArray(),
                "col" => 4
            ],
            [
                "type" => "select",
                "name" => "id_sales",
                "label" => "Sales",
                "options" => User::role('SALES')->orderBy('name')->pluck('name', 'id')->toArray(),
                "col" => 4
            ],
            [
                "type" => "select",
                "name" => "status",
                "label" => "Status",
                "options" => [
                    "pending" => "Pending",
                    "approved" => "Approved",
                    "rejected" => "Rejected",
                ],
                "col" => 4
            ],
            [
                "type" => "date",
                "name" => "invoice_date",
                "label" => "Invoice Date",
                "col" => 4
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Transaction::with(['customer', 'sales', 'parentTransaction'])
            ->where('type', 'salesorder');

        $actions = $this->addPermissionsToActions($this->getActions(), 'sales-order');
        $result = $this->processDataTable(
            $request,
            $query,
            $this->getColumns(),
            $actions,
            $this->getFilters(),
            $this->getFormatters()
        );

        return response()->json($result);
    }

    /**
     * Show the form for creating a new sales order from quotation.
     */
    public function create(Request $request)
    {
        $quotationId = $request->get('quotation_id');

        if (!$quotationId) {
            return redirect()
                ->route('quotation.index')
                ->with('error', 'Quotation ID is required to create a sales order.');
        }

        $quotation = Transaction::with(['customer', 'sales', 'details.origin', 'details.destination', 'details.loadout'])
            ->where('type', 'quotation')
            ->findOrFail($quotationId);

        // Check if quotation can be approved
        if ($quotation->status !== 'pending') {
            return redirect()
                ->route('quotation.show', $quotationId)
                ->with('error', 'Only pending quotations can be approved.');
        }

        // Check if sales order already exists for this quotation
        $existingSalesOrder = Transaction::where('type', 'salesorder')
            ->where('id_transaction', $quotationId)
            ->first();

        if ($existingSalesOrder) {
            return redirect()
                ->route('sales-order.show', $existingSalesOrder->id)
                ->with('info', 'Sales order already exists for this quotation.');
        }

        $model = new Transaction();
        $customers = Customer::orderBy('full_name')->get();
        $routes = Route::with(['origin', 'destination'])->get();
        $locations = Location::orderBy('name')->get();
        $loadouts = TruckLoadout::orderBy('name')->get();

        // Get PPH and PPN percentages from config
        $pphConfig = Config::where('key', 'PPH')->first();
        $ppnConfig = Config::where('key', 'PPN')->first();

        $pphPercentage = $pphConfig ? (float) $pphConfig->value : 2;
        $ppnPercentage = $ppnConfig ? (float) $ppnConfig->value : 11;

        // Check current user's role
        $currentUser = Auth::user();
        $currentUserRole = $currentUser->role ? $currentUser->role->name : null;
        $isNotSales = $currentUserRole !== 'SALES';

        // Get users with SALES role if current user is not SALES
        $salesUsers = collect();
        if ($isNotSales) {
            $salesRole = Role::where('name', 'SALES')->first();
            if ($salesRole) {
                $salesUsers = User::role('SALES')->orderBy('name')->get();
            }
        }

        return view("sales-order.form", [
            "model" => $model,
            "quotation" => $quotation,
            "customers" => $customers,
            "routes" => $routes,
            "locations" => $locations,
            "loadouts" => $loadouts,
            "pphPercentage" => $pphPercentage,
            "ppnPercentage" => $ppnPercentage,
            "isNotSales" => $isNotSales,
            "salesUsers" => $salesUsers,
            "isBackUrl" => route("quotation.show", $quotationId)
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(SalesOrderRequest $request)
    {
        try {
            $validated = $request->validated();

            // Get quotation
            $quotationId = $request->input('quotation_id');
            $quotation = Transaction::where('type', 'quotation')->findOrFail($quotationId);

            // Check if quotation can be approved
            if ($quotation->status !== 'pending') {
                return redirect()
                    ->route('quotation.show', $quotationId)
                    ->with('error', 'Only pending quotations can be approved.');
            }

            // Get customer to check tax settings
            $customer = Customer::findOrFail($validated['id_customer']);

            // Get PPH and PPN percentages from config
            $pphConfig = Config::where('key', 'PPH')->first();
            $ppnConfig = Config::where('key', 'PPN')->first();
            $pphPercentage = $pphConfig ? (float) $pphConfig->value : 2;
            $ppnPercentage = $ppnConfig ? (float) $ppnConfig->value : 11;

            // Calculate sub_total for each detail (price * quantity)
            $total = 0;
            foreach ($validated['details'] as $key => $detail) {
                $subtotal = $detail['price'] * $detail['quantity'];
                $validated['details'][$key]['sub_total'] = $subtotal;
                $total += $subtotal;
            }

            // Get discount
            $discount = $validated['discount'] ?? 0;
            $afterDiscount = $total - $discount;

            // Calculate PPN and PPH based on customer settings
            $ppn = 0;
            $pph = 0;
            if ($customer->is_ppn) {
                $ppn = $afterDiscount * ($ppnPercentage / 100);
            }
            if ($customer->is_pph) {
                $pph = $afterDiscount * ($pphPercentage / 100);
            }

            // Calculate grand total
            $grandTotal = $afterDiscount + $ppn - $pph;

            // Add calculated values to validated data
            $validated['total'] = $total;
            $validated['ppn'] = $ppn;
            $validated['ppn_percentage'] = $ppnPercentage;
            $validated['pph'] = $pph;
            $validated['pph_percentage'] = $pphPercentage;
            $validated['grand_total'] = $grandTotal;

            // Create sales order and update quotation status
            $salesOrder = $this->salesOrderRepository->createSalesOrder($validated, $quotation);

            return redirect()
                ->route('sales-order.show', $salesOrder->id)
                ->with('success', 'Sales order created successfully and quotation has been approved.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to create sales order');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->salesOrderRepository->findOrFail($id);
        $model->load(['createdBy', 'updatedBy']);

        return view("sales-order.show", [
            "model" => $model,
            "isBackUrl" => route("sales-order.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->salesOrderRepository->findOrFail($id);
        $model->load('details.origin', 'details.destination', 'details.loadout', 'parentTransaction');

        // Only allow editing pending sales orders
        if ($model->status !== 'pending') {
            return redirect()
                ->route('sales-order.show', $id)
                ->with('error', 'Only pending sales orders can be edited.');
        }

        $customers = Customer::orderBy('full_name')->get();
        $routes = Route::with(['origin', 'destination'])->get();
        $locations = Location::orderBy('name')->get();
        $loadouts = TruckLoadout::orderBy('name')->get();

        // Get PPH and PPN percentages from config
        $pphConfig = Config::where('key', 'PPH')->first();
        $ppnConfig = Config::where('key', 'PPN')->first();

        $pphPercentage = $pphConfig ? (float) $pphConfig->value : 2;
        $ppnPercentage = $ppnConfig ? (float) $ppnConfig->value : 11;

        // Check current user's role
        $currentUser = Auth::user();
        $currentUserRole = $currentUser->role ? $currentUser->role->name : null;
        $isNotSales = $currentUserRole !== 'SALES';

        // Get users with SALES role if current user is not SALES
        $salesUsers = collect();
        if ($isNotSales) {
            $salesRole = Role::where('name', 'SALES')->first();
            if ($salesRole) {
                $salesUsers = User::role('SALES')->orderBy('name')->get();
            }
        }

        return view("sales-order.form", [
            "model" => $model,
            "quotation" => $model->parentTransaction,
            "customers" => $customers,
            "routes" => $routes,
            "locations" => $locations,
            "loadouts" => $loadouts,
            "pphPercentage" => $pphPercentage,
            "ppnPercentage" => $ppnPercentage,
            "isNotSales" => $isNotSales,
            "salesUsers" => $salesUsers,
            "isBackUrl" => route("sales-order.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(SalesOrderRequest $request, string $id)
    {
        try {
            $model = $this->salesOrderRepository->findOrFail($id);

            // Only allow updating pending sales orders
            if ($model->status !== 'pending') {
                return redirect()
                    ->route('sales-order.show', $id)
                    ->with('error', 'Only pending sales orders can be updated.');
            }

            $validated = $request->validated();

            // Get customer to check tax settings
            $customer = Customer::findOrFail($validated['id_customer']);

            // Get PPH and PPN percentages from config
            $pphConfig = Config::where('key', 'PPH')->first();
            $ppnConfig = Config::where('key', 'PPN')->first();
            $pphPercentage = $pphConfig ? (float) $pphConfig->value : 2;
            $ppnPercentage = $ppnConfig ? (float) $ppnConfig->value : 11;

            // Calculate sub_total for each detail (price * quantity)
            $total = 0;
            foreach ($validated['details'] as $key => $detail) {
                $subtotal = $detail['price'] * $detail['quantity'];
                $validated['details'][$key]['sub_total'] = $subtotal;
                $total += $subtotal;
            }

            // Get discount
            $discount = $validated['discount'] ?? 0;
            $afterDiscount = $total - $discount;

            // Calculate PPN and PPH based on customer settings
            $ppn = 0;
            $pph = 0;
            if ($customer->is_ppn) {
                $ppn = $afterDiscount * ($ppnPercentage / 100);
            }
            if ($customer->is_pph) {
                $pph = $afterDiscount * ($pphPercentage / 100);
            }

            // Calculate grand total
            $grandTotal = $afterDiscount + $ppn - $pph;

            // Add calculated values to validated data
            $validated['total'] = $total;
            $validated['ppn'] = $ppn;
            $validated['ppn_percentage'] = $ppnPercentage;
            $validated['pph'] = $pph;
            $validated['pph_percentage'] = $pphPercentage;
            $validated['grand_total'] = $grandTotal;

            // Update sales order
            $salesOrder = $this->salesOrderRepository->updateSalesOrder($id, $validated);

            return redirect()
                ->route('sales-order.show', $salesOrder->id)
                ->with('success', 'Sales order updated successfully.');
        } catch (\Exception $e) {
            return $this->handleException($e, 'Failed to update sales order');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $model = $this->salesOrderRepository->findOrFail($id);

            // Only allow deleting pending sales orders
            if ($model->status !== 'pending') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only pending sales orders can be deleted.'
                ], 400);
            }

            // Revert quotation status back to pending
            if ($model->parentTransaction) {
                $model->parentTransaction->update([
                    'status' => 'pending',
                    'updated_by' => Auth::id(),
                ]);
            }

            $this->salesOrderRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Sales order deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete sales order');
        }
    }

    /**
     * Get prices for a specific route and customer
     */
    public function getPrices(Request $request)
    {
        $routeId = $request->input('route_id');
        $customerId = $request->input('customer_id');

        $prices = Price::with(['route', 'loadout', 'customer'])
            ->where('id_route', $routeId)
            ->where(function ($query) use ($customerId) {
                $query->where('id_customer', $customerId)
                    ->orWhereNull('id_customer');
            })
            ->get();

        return response()->json([
            'success' => true,
            'data' => $prices
        ]);
    }

    public function getRouteAllowances(Request $request)
    {
        $routeId = $request->input('route_id');

        $route = Route::find($routeId);

        if (!$route) {
            return response()->json([
                'success' => false,
                'message' => 'Route not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'pocket_money_1' => $route->pocket_money_1 ?? 0,
                'pocket_money_2' => $route->pocket_money_2 ?? 0,
                'pocket_money_3' => $route->pocket_money_3 ?? 0,
                'bonus' => $route->bonus ?? 0,
            ]
        ]);
    }
}
