<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\TruckRequest;
use App\Models\Truck;
use App\Models\TruckType;
use App\Repositories\TruckRepository;
use App\Repositories\TruckTypeRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class TruckController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $truckRepository;
    protected $truckTypeRepository;

    public function __construct(TruckRepository $truckRepository, TruckTypeRepository $truckTypeRepository)
    {
        $this->truckRepository = $truckRepository;
        $this->truckTypeRepository = $truckTypeRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:truck.index', only: ['index']),
            new Middleware('permission:truck.show', only: ['show']),
            new Middleware('permission:truck.store', only: ['create', 'store']),
            new Middleware('permission:truck.update', only: ['edit', 'update']),
            new Middleware('permission:truck.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 4,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "license_plate", "title" => "License Plate", "orderable" => true, "className" => "text-start"],
            ["data" => "id_type", "title" => "Truck Type", "orderable" => true, "relation" => "type", "relation_column" => "name", "className" => "text-start"],
            ["data" => "tax_date", "title" => "Tax Date", "orderable" => true, "className" => "text-start"],
            ["data" => "kir_date", "title" => "KIR Date", "orderable" => true, "className" => "text-start"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "truck.show"],
            ["type" => "edit", "url" => "truck.edit"],
            ["type" => "delete", "url" => "truck.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "select",
                "name" => "id_type",
                "label" => "Truck Type",
                "options" => Helpers::toDropdownOptions($this->truckTypeRepository->all(), "name", "id"),
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "license_plate",
                "label" => "License Plate",
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "tax_date",
                "label" => "Tax Date",
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "kir_date",
                "label" => "KIR Date",
                "col" => 3
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = Truck::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'truck');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("truck.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new Truck();
        $truckTypes = $this->truckTypeRepository->all();

        return view("truck.form", [
            "model" => $model,
            "truckTypes" => $truckTypes,
            "isBackUrl" => route("truck.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TruckRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();

            // Handle file uploads
            if ($request->hasFile('kir_file')) {
                $data['kir_file'] = $request->file('kir_file')->store('trucks/kir', 'public');
            }
            if ($request->hasFile('bpkb_file')) {
                $data['bpkb_file'] = $request->file('bpkb_file')->store('trucks/bpkb', 'public');
            }
            if ($request->hasFile('stnk_file')) {
                $data['stnk_file'] = $request->file('stnk_file')->store('trucks/stnk', 'public');
            }

            $this->truckRepository->create($data);

            DB::commit();

            return redirect()
                ->route('truck.index')
                ->with('success', 'Truck created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create truck');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->truckRepository->findOrFail($id);

        return view("truck.show", [
            "model" => $model,
            "isBackUrl" => route("truck.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->truckRepository->findOrFail($id);
        $truckTypes = $this->truckTypeRepository->all();

        return view("truck.form", [
            "model" => $model,
            "truckTypes" => $truckTypes,
            "isBackUrl" => route("truck.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(TruckRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $truck = $this->truckRepository->findOrFail($id);
            $data = $request->validated();

            // Handle file uploads and delete old files
            if ($request->hasFile('kir_file')) {
                if ($truck->kir_file && Storage::disk('public')->exists($truck->kir_file)) {
                    Storage::disk('public')->delete($truck->kir_file);
                }
                $data['kir_file'] = $request->file('kir_file')->store('trucks/kir', 'public');
            }
            if ($request->hasFile('bpkb_file')) {
                if ($truck->bpkb_file && Storage::disk('public')->exists($truck->bpkb_file)) {
                    Storage::disk('public')->delete($truck->bpkb_file);
                }
                $data['bpkb_file'] = $request->file('bpkb_file')->store('trucks/bpkb', 'public');
            }
            if ($request->hasFile('stnk_file')) {
                if ($truck->stnk_file && Storage::disk('public')->exists($truck->stnk_file)) {
                    Storage::disk('public')->delete($truck->stnk_file);
                }
                $data['stnk_file'] = $request->file('stnk_file')->store('trucks/stnk', 'public');
            }

            $this->truckRepository->update($id, $data);

            DB::commit();

            return redirect()
                ->route('truck.index')
                ->with('success', 'Truck updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update truck');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->truckRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Truck deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete truck');
        }
    }
}
