<?php

namespace App\Http\Controllers;

use App\Http\Requests\TruckLoadoutRequest;
use App\Models\TruckLoadout;
use App\Repositories\TruckLoadoutRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class TruckLoadoutController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $truckLoadoutRepository;

    public function __construct(TruckLoadoutRepository $truckLoadoutRepository)
    {
        $this->truckLoadoutRepository = $truckLoadoutRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:truck-loadout.index', only: ['index']),
            new Middleware('permission:truck-loadout.show', only: ['show']),
            new Middleware('permission:truck-loadout.store', only: ['create', 'store']),
            new Middleware('permission:truck-loadout.update', only: ['edit', 'update']),
            new Middleware('permission:truck-loadout.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 1,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "name", "title" => "Loadout Name", "orderable" => true, "className" => "text-start"],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, "className" => "text-end"],
            ["data" => "action", "title" => "", "orderable" => false, "className" => "action"],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "truck-loadout.show"],
            ["type" => "edit", "url" => "truck-loadout.edit"],
            ["type" => "delete", "url" => "truck-loadout.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "name",
                "label" => "Loadout Name",
                "col" => 6
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = TruckLoadout::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'truck-loadout');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("truck-loadout.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new TruckLoadout();

        return view("truck-loadout.form", [
            "model" => $model,
            "isBackUrl" => route("truck-loadout.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TruckLoadoutRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->truckLoadoutRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('truck-loadout.index')
                ->with('success', 'Truck loadout created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create truck loadout');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->truckLoadoutRepository->findOrFail($id);

        return view("truck-loadout.show", [
            "model" => $model,
            "isBackUrl" => route("truck-loadout.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->truckLoadoutRepository->findOrFail($id);

        return view("truck-loadout.form", [
            "model" => $model,
            "isBackUrl" => route("truck-loadout.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(TruckLoadoutRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->truckLoadoutRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('truck-loadout.index')
                ->with('success', 'Truck loadout updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update truck loadout');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->truckLoadoutRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Truck loadout deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete truck loadout');
        }
    }
}
