<?php

namespace App\Http\Controllers;

use App\Http\Requests\TruckTypeRequest;
use App\Models\TruckType;
use App\Repositories\TruckTypeRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class TruckTypeController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $truckTypeRepository;

    public function __construct(TruckTypeRepository $truckTypeRepository)
    {
        $this->truckTypeRepository = $truckTypeRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:truck-type.index', only: ['index']),
            new Middleware('permission:truck-type.show', only: ['show']),
            new Middleware('permission:truck-type.store', only: ['create', 'store']),
            new Middleware('permission:truck-type.update', only: ['edit', 'update']),
            new Middleware('permission:truck-type.destroy', only: ['destroy']),
        ];
    }

    private $defaultOrder = [
        "order_by" => 1,
        "order_mode" => "desc"
    ];

    private $columns = [
        ["data" => "name", "title" => "Name", "orderable" => true, 'className' => 'text-start'],
        ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
        ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
    ];

    private $actions = [
        ["type" => "view", "url" => "truck-type.show"],
        ["type" => "edit", "url" => "truck-type.edit"],
        ["type" => "delete", "url" => "truck-type.destroy"],
    ];

    private $filters = [
        [
            "type" => "text",
            "name" => "name",
            "label" => "Name",
            "col" => 12
        ],
    ];

    public function getData(Request $request)
    {
        $query = TruckType::query();

        $actions = $this->addPermissionsToActions($this->actions, 'truck-type');
        $result = $this->processDataTable($request, $query, $this->columns, $actions, $this->filters);

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("truck-type.index", [
            "defaultOrder" => $this->defaultOrder,
            "columns" => $this->columns,
            "filters" => $this->filters
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new TruckType();

        return view("truck-type.form", [
            "model" => $model,
            "isBackUrl" => route("truck-type.index")
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TruckTypeRequest $request)
    {
        DB::beginTransaction();

        try {

            $this->truckTypeRepository->create($request->validated());

            DB::commit();

            return redirect()
                ->route('truck-type.index')
                ->with('success', 'Truck type created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create truck type');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->truckTypeRepository->findOrFail($id);

        return view("truck-type.show", [
            "model" => $model,
            "isBackUrl" => route("truck-type.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->truckTypeRepository->findOrFail($id);

        return view("truck-type.form", [
            "model" => $model,
            "isBackUrl" => route("truck-type.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(TruckTypeRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->truckTypeRepository->update($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('truck-type.index')
                ->with('success', 'Truck type updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update truck type');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->truckTypeRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Truck type deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete truck type');
        }
    }
}
