<?php

namespace App\Http\Controllers;

use App\Helpers\Helpers;
use App\Http\Requests\UserRequest;
use App\Models\User;
use App\Repositories\UserRepository;
use App\Repositories\RoleRepository;
use App\Traits\DataTableTrait;
use App\Traits\HandlesExceptions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class UserController implements HasMiddleware
{
    use DataTableTrait, HandlesExceptions;

    protected $userRepository;
    protected $roleRepository;

    public function __construct(UserRepository $userRepository, RoleRepository $roleRepository)
    {
        $this->userRepository = $userRepository;
        $this->roleRepository = $roleRepository;
    }

    public static function middleware(): array
    {
        return [
            new Middleware('permission:user.index', only: ['index']),
            new Middleware('permission:user.show', only: ['show']),
            new Middleware('permission:user.store', only: ['create', 'store']),
            new Middleware('permission:user.update', only: ['edit', 'update']),
            new Middleware('permission:user.destroy', only: ['destroy']),
        ];
    }

    private function getDefaultOrder()
    {
        return [
            "order_by" => 5,
            "order_mode" => "desc"
        ];
    }

    private function getColumns()
    {
        return [
            ["data" => "name", "title" => "Name", "orderable" => true, 'className' => 'text-start'],
            ["data" => "username", "title" => "Username", "orderable" => true, 'className' => 'text-start'],
            ["data" => "role", "title" => "Role", "orderable" => true, 'className' => 'text-start', "relation" => "role", "relation_column" => "name", "exclude_column" => true],
            ["data" => "created_at", "title" => "Created At", "orderable" => true, 'className' => 'text-end'],
            ["data" => "action", "title" => "", "orderable" => false, 'className' => 'action'],
        ];
    }

    private function getActions()
    {
        return [
            ["type" => "view", "url" => "user.show"],
            ["type" => "edit", "url" => "user.edit"],
            ["type" => "delete", "url" => "user.destroy"],
        ];
    }

    private function getFilters()
    {
        return [
            [
                "type" => "text",
                "name" => "name",
                "label" => "Name",
                "col" => 3
            ],
            [
                "type" => "text",
                "name" => "username",
                "label" => "Username",
                "col" => 3
            ],
            [
                "type" => "select",
                "options" => Helpers::toDropdownOptions(Role::all(), "name", "id"),
                "name" => "role",
                "label" => "Role",
                "col" => 3,
                'relation' => 'roles',
                'relation_column' => 'id'
            ],
            [
                "type" => "text",
                "name" => "address",
                "label" => "Address",
                "col" => 3
            ],
        ];
    }

    public function getData(Request $request)
    {
        $query = User::query();

        $actions = $this->addPermissionsToActions($this->getActions(), 'user');
        $result = $this->processDataTable($request, $query, $this->getColumns(), $actions, $this->getFilters());

        return response()->json($result);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("user.index", [
            "defaultOrder" => $this->getDefaultOrder(),
            "columns" => $this->getColumns(),
            "filters" => $this->getFilters()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $model = new User();
        $roles = $this->roleRepository->all();

        return view("user.form", [
            "model" => $model,
            "roles" => $roles,
            "isBackUrl" => route("user.index")
        ]);
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(UserRequest $request)
    {
        DB::beginTransaction();

        try {
            $this->userRepository->createUser($request->validated());

            DB::commit();

            return redirect()
                ->route('user.index')
                ->with('success', 'User created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to create user');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $model = $this->userRepository->findOrFail($id);

        return view("user.show", [
            "model" => $model,
            "isBackUrl" => route("user.index")
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $model = $this->userRepository->findOrFail($id);
        $roles = $this->roleRepository->all();

        return view("user.form", [
            "model" => $model,
            "roles" => $roles,
            "isBackUrl" => route("user.index")
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UserRequest $request, string $id)
    {
        DB::beginTransaction();

        try {
            $this->userRepository->updateUser($id, $request->validated());

            DB::commit();

            return redirect()
                ->route('user.index')
                ->with('success', 'User updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleException($e, 'Failed to update user');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        DB::beginTransaction();

        try {
            $this->userRepository->softDelete($id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'User deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->handleJsonException($e, 'Failed to delete user');
        }
    }
}
