<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class AccountRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation()
    {
        if ($this->has('nature')) {
            $this->merge([
                'nature' => strtolower($this->nature),
            ]);
        }
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $accountId = $this->route('account');

        return [
            'code' => [
                'required',
                'string',
                'max:50',
                Rule::unique('accounts', 'code')->ignore($accountId)->whereNull('deleted_at'),
            ],
            'name' => 'required|string|max:255',
            'id_category' => 'required|exists:account_categories,id',
            'id_type' => 'required|exists:account_types,id',
            'nature' => 'required|in:debit,credit',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'code' => 'account code',
            'name' => 'account name',
            'id_category' => 'category',
            'id_type' => 'type',
            'nature' => 'nature',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'code.required' => 'The account code field is required.',
            'code.unique' => 'An account with this code already exists.',
            'name.required' => 'The account name field is required.',
            'id_category.required' => 'The category field is required.',
            'id_category.exists' => 'The selected category is invalid.',
            'id_type.required' => 'The type field is required.',
            'id_type.exists' => 'The selected type is invalid.',
            'nature.required' => 'The nature field is required.',
            'nature.in' => 'The selected nature is invalid. Valid options are: debit or credit.',
        ];
    }
}
