<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class CustomerRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $customerId = $this->route('customer');

        return [
            'full_name' => 'required|string|max:255',
            'email' => [
                'nullable',
                'email',
                'max:45',
                Rule::unique('customers', 'email')->ignore($customerId)->whereNull('deleted_at'),
            ],
            'phone' => 'required|string|max:16',
            'address' => 'required|string|max:255',
            'pic_name' => 'required|string|max:255',
            'pic_phone' => 'required|string|max:45',
            'term_of_payment' => 'required|max:45',
            'credit_limit' => 'required|numeric|min:0',
            'is_ppn' => 'nullable|boolean',
            'is_pph' => 'nullable|boolean',
            'npwp_no' => 'nullable|string|max:16',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'full_name' => 'full name',
            'email' => 'email address',
            'phone' => 'phone number',
            'address' => 'address',
            'pic_name' => 'PIC name',
            'pic_phone' => 'PIC phone',
            'term_of_payment' => 'payment terms',
            'credit_limit' => 'credit limit',
            'is_ppn' => 'PPN',
            'is_pph' => 'PPh',
            'npwp_no' => 'NPWP number',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'full_name.required' => 'The full name field is required.',
            'email.email' => 'Please enter a valid email address.',
            'email.unique' => 'A customer with this email already exists.',
            'phone.required' => 'The phone number field is required.',
            'address.required' => 'The address field is required.',
            'pic_name.required' => 'The PIC name field is required.',
            'pic_phone.required' => 'The PIC phone field is required.',
            'term_of_payment.required' => 'The payment terms field is required.',
            'credit_limit.required' => 'The credit limit field is required.',
            'credit_limit.numeric' => 'The credit limit must be a number.',
            'credit_limit.min' => 'The credit limit must be at least 0.',
            'is_ppn.required' => 'The PPN field is required.',
            'is_pph.required' => 'The PPh field is required.',
            'npwp_no.max' => 'The NPWP number may not be greater than 16 characters.',
        ];
    }

    /**
     * Prepare the data for validation.
     * Remove thousand separators from formatted numbers
     */
    protected function prepareForValidation(): void
    {
        $data = [];

        if ($this->has('term_of_payment') && $this->term_of_payment !== null && $this->term_of_payment !== '') {
            $data['term_of_payment'] = \App\Helpers\Helpers::parser($this->term_of_payment);
        }

        if ($this->has('credit_limit') && $this->credit_limit !== null && $this->credit_limit !== '') {
            $data['credit_limit'] = \App\Helpers\Helpers::parser($this->credit_limit);
        }

        if (!empty($data)) {
            $this->merge($data);
        }
    }
}
