<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;
use App\Models\TransactionDetail;

class DeliveryOrderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'id_transaction' => 'required|exists:transactions,id',
            'details' => 'required|array|min:1',
            'details.*.id' => 'nullable|exists:deliver_order_details,id',
            'details.*.id_detail' => 'required|exists:transaction_details,id',
            'details.*.id_driver' => 'required|exists:drivers,id',
            'details.*.id_truck' => 'required|exists:trucks,id',
            'details.*.qty' => 'required|integer|min:1',
            'details.*.delivery_date' => 'required|date',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'id_transaction' => 'sales order',
            'details' => 'delivery order details',
            'details.*.id_detail' => 'route detail',
            'details.*.id_driver' => 'driver',
            'details.*.id_truck' => 'truck',
            'details.*.qty' => 'quantity',
            'details.*.delivery_date' => 'delivery date',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'details.required' => 'At least one delivery order detail is required.',
            'details.min' => 'At least one delivery order detail is required.',
        ];
    }

    /**
     * Configure the validator instance.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $this->validateQtyAssignments($validator);
        });
    }

    /**
     * Validate that qty assignments match route quantities exactly.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    protected function validateQtyAssignments($validator)
    {
        $details = $this->input('details', []);

        // Debug: Dump all details
        // dd($details);

        // Group details by route (id_detail)
        $qtyByRoute = [];
        foreach ($details as $key => $detail) {
            $routeId = $detail['id_detail'] ?? null;
            $qty = $detail['qty'] ?? 0;

            if ($routeId) {
                if (!isset($qtyByRoute[$routeId])) {
                    $qtyByRoute[$routeId] = 0;
                }
                $qtyByRoute[$routeId] += $qty;
            }
        }

        // Debug: Dump grouped quantities by route
        // dd($qtyByRoute);

        // Validate each route's total qty
        foreach ($qtyByRoute as $routeId => $totalAssigned) {
            $transactionDetail = TransactionDetail::find($routeId);

            if (!$transactionDetail) {
                $validator->errors()->add(
                    'details',
                    "Route detail with ID {$routeId} not found."
                );
                continue;
            }

            $expectedQty = $transactionDetail->quantity;
            $routeName = ($transactionDetail->origin->name ?? '') . ' → ' . ($transactionDetail->destination->name ?? '');

            // Debug: Dump comparison data
            // dd([
            //     'route_id' => $routeId,
            //     'route_name' => $routeName,
            //     'expected_qty' => $expectedQty,
            //     'total_assigned' => $totalAssigned,
            //     'validator_errors' => $validator->errors()->toArray()
            // ]);

            if ($totalAssigned < $expectedQty) {
                $validator->errors()->add(
                    'details',
                    "Route \"{$routeName}\": Only {$totalAssigned} of {$expectedQty} units assigned. Please assign all {$expectedQty} units."
                );
            } elseif ($totalAssigned > $expectedQty) {
                $validator->errors()->add(
                    'details',
                    "Route \"{$routeName}\": {$totalAssigned} units assigned exceeds the available {$expectedQty} units."
                );
            }
        }   

        // Debug: Dump final validation errors
        // dd($validator->errors()->toArray());
    }
}
