<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class DriverRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $driverId = $this->route('driver');
        $isUpdate = $this->isMethod('put') || $this->isMethod('patch');

        return [
            'code' => [
                'required',
                'string',
                'max:50',
                Rule::unique('drivers', 'code')->ignore($driverId)->whereNull('deleted_at'),
            ],
            'full_name' => 'required|string|max:255',
            'nick_name' => 'nullable|string|max:255',
            'phone' => 'required|string|max:16',
            'address' => 'required|string|max:255',
            'identity_number' => 'required|string|max:16',
            'license_number' => [
                'required',
                'string',
                'max:16',
                Rule::unique('drivers', 'license_number')->ignore($driverId)->whereNull('deleted_at'),
            ],
            'kk_number' => 'nullable|string|max:16',
            'bpjs_number' => 'nullable|string|max:16',
            'identity_file' => ($isUpdate ? 'nullable' : 'required') . '|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'license_file' => ($isUpdate ? 'nullable' : 'required') . '|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'license_type' => 'required|in:B1,B1 UMUM,B2,B2 UMUM',
            'license_date' => 'required|date',
            'kk_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'photo_file' => ($isUpdate ? 'nullable' : 'required') . '|file|mimes:jpg,jpeg,png|max:2048',
            'skck_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'bank_name' => 'nullable|string|max:45',
            'bank_owner_name' => 'nullable|string|max:45',
            'bank_no' => 'nullable|string|max:45',
            'joined_date' => 'required|date',
            'status' => 'required|in:active,blacklist,resigned',
            'notes' => 'nullable|string|max:512',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'code' => 'driver code',
            'full_name' => 'full name',
            'nick_name' => 'nickname',
            'phone' => 'phone number',
            'address' => 'address',
            'identity_number' => 'identity number (KTP)',
            'license_number' => 'license number (SIM)',
            'kk_number' => 'family card number (KK)',
            'bpjs_number' => 'BPJS number',
            'license_type' => 'license type',
            'license_date' => 'license expiry date',
            'bank_name' => 'bank name',
            'bank_owner_name' => 'account holder name',
            'bank_no' => 'account number',
            'joined_date' => 'join date',
            'status' => 'status',
            'notes' => 'notes',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'code.required' => 'The driver code field is required.',
            'code.unique' => 'A driver with this code already exists.',
            'full_name.required' => 'The full name field is required.',
            'phone.required' => 'The phone number field is required.',
            'address.required' => 'The address field is required.',
            'identity_number.required' => 'The identity number (KTP) field is required.',
            'license_number.required' => 'The license number field is required.',
            'license_number.unique' => 'A driver with this license number already exists.',
            'identity_file.required' => 'The identity file field is required.',
            'license_file.required' => 'The license file field is required.',
            'license_type.required' => 'The license type field is required.',
            'license_type.in' => 'The selected license type is invalid.',
            'license_date.required' => 'The license expiry date field is required.',
            'photo_file.required' => 'The photo file field is required.',
            'joined_date.required' => 'The join date field is required.',
            'status.required' => 'The status field is required.',
            'status.in' => 'The selected status is invalid.',
        ];
    }
}
