<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;

class QuotationRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        // Check if current user is SALES
        $currentUser = Auth::user();
        $currentUserRole = $currentUser->role ? $currentUser->role->name : null;
        $isSales = $currentUserRole === 'SALES';

        return [
            'id_customer' => 'required|exists:customers,id',
            'id_sales' => $isSales ? 'nullable|exists:users,id' : 'required|exists:users,id',
            'invoice_date' => 'required|date',
            'notes' => 'nullable|string',
            'discount' => 'nullable|numeric|min:0',
            'ppn' => 'nullable|numeric|min:0',
            'pph' => 'nullable|numeric|min:0',
            'details' => 'required|array|min:1',
            'details.*.id' => 'nullable|exists:transaction_details,id',
            'details.*.id_origin' => 'required|exists:locations,id',
            'details.*.id_destination' => 'required|exists:locations,id',
            'details.*.id_loadout' => 'required|exists:truck_loadouts,id',
            'details.*.price' => 'required|numeric|min:0',
            'details.*.quantity' => 'required|numeric|min:1',
            'details.*.pocket_money_1' => 'required|numeric|min:0',
            'details.*.pocket_money_2' => 'nullable|numeric|min:0',
            'details.*.pocket_money_3' => 'nullable|numeric|min:0',
            'details.*.bonus' => 'nullable|numeric|min:0',
            'details.*.origin_action' => 'required|in:load,unload',
            'details.*.destination_action' => 'required|in:load,unload',
            'details.*.notes' => 'nullable|string',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'id_customer' => 'customer',
            'id_sales' => 'sales',
            'invoice_date' => 'invoice date',
            'notes' => 'notes',
            'discount' => 'discount',
            'ppn' => 'PPN',
            'pph' => 'PPh',
            'details' => 'transaction details',
            'details.*.id_origin' => 'origin',
            'details.*.id_destination' => 'destination',
            'details.*.id_loadout' => 'loadout',
            'details.*.price' => 'price',
            'details.*.quantity' => 'quantity',
            'details.*.pocket_money_1' => 'pocket money 1',
            'details.*.pocket_money_2' => 'pocket money 2',
            'details.*.pocket_money_3' => 'pocket money 3',
            'details.*.bonus' => 'bonus',
            'details.*.origin_action' => 'origin action',
            'details.*.destination_action' => 'destination action',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'id_customer.required' => 'Please select a customer.',
            'id_customer.exists' => 'The selected customer does not exist.',
            'id_sales.required' => 'Please select a sales person.',
            'id_sales.exists' => 'The selected sales does not exist.',
            'invoice_date.required' => 'The invoice date field is required.',
            'invoice_date.date' => 'Please enter a valid invoice date.',
            'discount.numeric' => 'The discount must be a number.',
            'discount.min' => 'The discount must be at least 0.',
            'ppn.numeric' => 'The PPN must be a number.',
            'ppn.min' => 'The PPN must be at least 0.',
            'pph.numeric' => 'The PPh must be a number.',
            'pph.min' => 'The PPh must be at least 0.',
            'details.required' => 'Please add at least one transaction detail.',
            'details.min' => 'Please add at least one transaction detail.',
            'details.*.id_origin.required' => 'The origin field is required for all details.',
            'details.*.id_origin.exists' => 'The selected origin does not exist.',
            'details.*.id_destination.required' => 'The destination field is required for all details.',
            'details.*.id_destination.exists' => 'The selected destination does not exist.',
            'details.*.id_loadout.required' => 'The loadout field is required for all details.',
            'details.*.id_loadout.exists' => 'The selected loadout does not exist.',
            'details.*.price.required' => 'The price field is required for all details.',
            'details.*.price.numeric' => 'The price must be a number.',
            'details.*.price.min' => 'The price must be at least 0.',
            'details.*.quantity.required' => 'The quantity field is required for all details.',
            'details.*.quantity.numeric' => 'The quantity must be a number.',
            'details.*.quantity.min' => 'The quantity must be at least 1.',
            'details.*.origin_action.required' => 'The origin action field is required for all details.',
            'details.*.origin_action.in' => 'The origin action must be either load or unload.',
            'details.*.destination_action.required' => 'The destination action field is required for all details.',
            'details.*.destination_action.in' => 'The destination action must be either load or unload.',
        ];
    }

    /**
     * Prepare the data for validation.
     * Remove thousand separators from formatted numbers
     */
    protected function prepareForValidation(): void
    {
        // Clean discount, ppn, pph fields
        $topLevelFields = ['discount', 'ppn', 'pph'];
        foreach ($topLevelFields as $field) {
            if ($this->has($field) && $this->$field !== null && $this->$field !== '') {
                $this->merge([$field => \App\Helpers\Helpers::parser($this->$field)]);
            }
        }

        // Clean number fields in details array
        if ($this->has('details') && is_array($this->details)) {
            $cleanedDetails = [];
            $numberFields = ['price', 'quantity', 'pocket_money_1', 'pocket_money_2', 'pocket_money_3', 'bonus'];

            foreach ($this->details as $index => $detail) {
                $cleanedDetail = $detail;

                foreach ($numberFields as $field) {
                    if (isset($detail[$field]) && $detail[$field] !== null && $detail[$field] !== '') {
                        $cleanedDetail[$field] = \App\Helpers\Helpers::parser($detail[$field]);
                    }
                }

                $cleanedDetails[$index] = $cleanedDetail;
            }

            $this->merge(['details' => $cleanedDetails]);
        }
    }
}
