<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class RouteRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'id_origin' => 'required|exists:locations,id',
            'id_destination' => 'required|exists:locations,id',
            'distance' => 'required|numeric|min:0',
            'duration' => 'required|integer|min:0',
            'toll' => 'required|in:non,full,half',
            'pocket_money_1' => 'required|numeric|min:0',
            'pocket_money_2' => 'nullable|numeric|min:0',
            'pocket_money_3' => 'nullable|numeric|min:0',
            'bonus' => 'nullable|numeric|min:0',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'id_origin' => 'origin location',
            'id_destination' => 'destination location',
            'distance' => 'distance',
            'duration' => 'duration',
            'toll' => 'toll fee',
            'pocket_money_1' => 'pocket money 1',
            'pocket_money_2' => 'pocket money 2',
            'pocket_money_3' => 'pocket money 3',
            'bonus' => 'bonus',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'id_origin.required' => 'The origin location field is required.',
            'id_origin.exists' => 'The selected origin location is invalid.',
            'id_destination.required' => 'The destination location field is required.',
            'id_destination.exists' => 'The selected destination location is invalid.',
            'distance.required' => 'The distance field is required.',
            'distance.numeric' => 'The distance must be a number.',
            'distance.min' => 'The distance must be at least 0.',
            'duration.required' => 'The duration field is required.',
            'duration.integer' => 'The duration must be a number.',
            'duration.min' => 'The duration must be at least 0.',
            'toll.required' => 'The toll field is required.',
            'toll.in' => 'The selected toll option is invalid. Valid options are: non, full, half.',
            'pocket_money_1.required' => 'The pocket money 1 field is required.',
            'pocket_money_1.numeric' => 'The pocket money 1 must be a number.',
            'pocket_money_1.min' => 'The pocket money 1 must be at least 0.',
        ];
    }

    /**
     * Prepare the data for validation.
     * Remove thousand separators from formatted numbers
     */
    protected function prepareForValidation(): void
    {
        $numberFields = ['distance', 'duration', 'pocket_money_1', 'pocket_money_2', 'pocket_money_3', 'bonus'];

        $cleaned = [];
        foreach ($numberFields as $field) {
            if ($this->has($field) && $this->$field !== null && $this->$field !== '') {
                $cleaned[$field] = \App\Helpers\Helpers::parser($this->$field);
            }
        }

        if (!empty($cleaned)) {
            $this->merge($cleaned);
        }
    }
}
