<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class SalesOrderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'id_customer' => 'required|exists:customers,id',
            'id_sales' => 'nullable|exists:users,id',
            'invoice_date' => 'required|date',
            'notes' => 'nullable|string',
            'discount' => 'nullable|numeric|min:0',
            'details' => 'required|array|min:1',
            'details.*.id' => 'nullable|exists:transaction_details,id',
            'details.*.id_origin' => 'required|exists:locations,id',
            'details.*.id_destination' => 'required|exists:locations,id',
            'details.*.id_loadout' => 'required|exists:truck_loadouts,id',
            'details.*.price' => 'required|numeric|min:0',
            'details.*.quantity' => 'required|numeric|min:1',
            'details.*.origin_action' => 'required|in:load,unload',
            'details.*.destination_action' => 'required|in:load,unload',
            'details.*.pocket_money_1' => 'nullable|numeric|min:0',
            'details.*.pocket_money_2' => 'nullable|numeric|min:0',
            'details.*.pocket_money_3' => 'nullable|numeric|min:0',
            'details.*.bonus' => 'nullable|numeric|min:0',
            'details.*.notes' => 'nullable|string',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'id_customer' => 'customer',
            'id_sales' => 'sales person',
            'invoice_date' => 'invoice date',
            'details' => 'sales order details',
            'details.*.id_origin' => 'origin location',
            'details.*.id_destination' => 'destination location',
            'details.*.id_loadout' => 'loadout',
            'details.*.price' => 'price',
            'details.*.quantity' => 'quantity',
            'details.*.origin_action' => 'origin action',
            'details.*.destination_action' => 'destination action',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'details.required' => 'At least one sales order detail is required.',
            'details.min' => 'At least one sales order detail is required.',
        ];
    }

    /**
     * Prepare the data for validation.
     * Remove thousand separators from formatted numbers
     */
    protected function prepareForValidation(): void
    {
        // Clean discount field
        if ($this->has('discount') && $this->discount !== null && $this->discount !== '') {
            $this->merge([
                'discount' => \App\Helpers\Helpers::parser($this->discount),
            ]);
        }

        // Clean number fields in details array
        if ($this->has('details') && is_array($this->details)) {
            $cleanedDetails = [];
            $numberFields = ['price', 'quantity', 'pocket_money_1', 'pocket_money_2', 'pocket_money_3', 'bonus'];

            foreach ($this->details as $index => $detail) {
                $cleanedDetail = $detail;

                foreach ($numberFields as $field) {
                    if (isset($detail[$field]) && $detail[$field] !== null && $detail[$field] !== '') {
                        $cleanedDetail[$field] = \App\Helpers\Helpers::parser($detail[$field]);
                    }
                }

                $cleanedDetails[$index] = $cleanedDetail;
            }

            $this->merge(['details' => $cleanedDetails]);
        }
    }
}
