<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreJournalRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'date' => 'required|date',
            'description' => 'nullable|string|max:255',
            'details' => 'required|array|min:1',
            'details.*.id_accounts' => 'required|exists:accounts,id',
            'details.*.debit' => 'required|numeric|min:0',
            'details.*.credit' => 'required|numeric|min:0',
            'details.*.description' => 'nullable|string',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages(): array
    {
        return [
            'details.required' => 'At least one account journal detail is required.',
            'details.*.id_accounts.required' => 'Account is required for each detail.',
            'details.*.id_accounts.exists' => 'Selected account is invalid.',
            'details.*.debit.required' => 'Debit amount is required.',
            'details.*.credit.required' => 'Credit amount is required.',
        ];
    }

    /**
     * Prepare the data for validation.
     * Remove thousand separators from formatted numbers
     */
    protected function prepareForValidation(): void
    {
        if ($this->has('details') && is_array($this->details)) {
            $cleanedDetails = [];

            foreach ($this->details as $index => $detail) {
                $cleanedDetail = $detail;

                if (isset($detail['debit']) && $detail['debit'] !== null && $detail['debit'] !== '') {
                    $cleanedDetail['debit'] = \App\Helpers\Helpers::parser($detail['debit']);
                }

                if (isset($detail['credit']) && $detail['credit'] !== null && $detail['credit'] !== '') {
                    $cleanedDetail['credit'] = \App\Helpers\Helpers::parser($detail['credit']);
                }

                $cleanedDetails[$index] = $cleanedDetail;
            }

            $this->merge(['details' => $cleanedDetails]);
        }
    }

    /**
     * Configure the validator instance.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $details = $this->input('details', []);
            $totalDebit = 0;
            $totalCredit = 0;

            foreach ($details as $detail) {
                $totalDebit += floatval($detail['debit'] ?? 0);
                $totalCredit += floatval($detail['credit'] ?? 0);
            }

            if (abs($totalDebit - $totalCredit) > 0.01) {
                $validator->errors()->add('details', 'Total debit and credit must be balanced.');
            }
        });
    }
}
