<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class TruckRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $truckId = $this->route('truck');

        return [
            'id_type' => 'required|exists:truck_types,id',
            'code' => [
                'nullable',
                'string',
                'max:16',
                Rule::unique('trucks', 'code')->ignore($truckId)->whereNull('deleted_at'),
            ],
            'brand' => 'nullable|string|max:64',
            'body_no' => 'nullable|string|max:45',
            'machine_no' => 'nullable|string|max:45',
            'vehicle_year' => 'nullable|integer|min:1900|max:' . (date('Y') + 1),
            'stnk_owner' => 'nullable|string|max:255',
            'kir_name' => 'nullable|string|max:255',
            'license_plate' => [
                'required',
                'string',
                'max:16',
                Rule::unique('trucks', 'license_plate')->ignore($truckId)->whereNull('deleted_at'),
            ],
            'tax_date' => 'nullable|date',
            'kir_date' => 'nullable|date',
            'stnk_date' => 'nullable|date',
            'kir_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'bpkb_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'stnk_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'credit_start_date' => 'nullable|date',
            'credit_end_date' => 'nullable|date|after_or_equal:credit_start_date',
            'cctv_gps_no' => 'nullable|string|max:32',
            'gps_contact' => 'nullable|string|max:16',
            'note' => 'nullable|string|max:255',
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array<string, string>
     */
    public function attributes(): array
    {
        return [
            'id_type' => 'truck type',
            'code' => 'truck code',
            'brand' => 'brand',
            'body_no' => 'body number',
            'machine_no' => 'machine number',
            'vehicle_year' => 'vehicle year',
            'stnk_owner' => 'STNK owner',
            'kir_name' => 'KIR name',
            'license_plate' => 'license plate',
            'tax_date' => 'tax date',
            'kir_date' => 'KIR date',
            'stnk_date' => 'STNK date',
            'credit_start_date' => 'credit start date',
            'credit_end_date' => 'credit end date',
            'cctv_gps_no' => 'CCTV/GPS number',
            'gps_contact' => 'GPS contact',
            'note' => 'note',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'id_type.required' => 'The truck type field is required.',
            'id_type.exists' => 'The selected truck type is invalid.',
            'code.unique' => 'A truck with this code already exists.',
            'license_plate.required' => 'The license plate field is required.',
            'license_plate.unique' => 'A truck with this license plate already exists.',
            'vehicle_year.integer' => 'The vehicle year must be a valid year.',
            'vehicle_year.min' => 'The vehicle year must be at least 1900.',
            'vehicle_year.max' => 'The vehicle year cannot be greater than next year.',
            'credit_end_date.after_or_equal' => 'The credit end date must be after or equal to the credit start date.',
        ];
    }
}
