<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\LogsActivityWithDetails;

class DeliverOrderDetail extends Model
{
    use SoftDeletes, LogsActivityWithDetails;

    protected $fillable = [
        'id_delivery_order',
        'id_detail',
        'id_driver',
        'id_truck',
        'delivery_date',
        'qty',
        'status',
        'driver_access_token',
        'token_expires_at',
        'created_by',
        'updated_by',
        'deleted_by',
    ];

    protected $casts = [
        'delivery_date' => 'date',
        'token_expires_at' => 'datetime',
        'created_at' => 'datetime',
    ];

    /**
     * Get attributes to log - exclude parent foreign key.
     */
    protected function getLogAttributes(): array
    {
        // Exclude parent foreign key: id_delivery_order (this is handled by parent observer)
        // Exclude auto-generated tokens: driver_access_token, token_expires_at
        return array_diff($this->fillable, ['id_delivery_order', 'driver_access_token', 'token_expires_at']);
    }

    /**
     * Get the options for logging activity.
     * Disable automatic logging for detail models - parent observer handles it.
     */
    public function getActivitylogOptions(): \Spatie\Activitylog\LogOptions
    {
        return \Spatie\Activitylog\LogOptions::defaults()
            ->logOnly([]) // Don't log anything automatically
            ->dontSubmitEmptyLogs();
    }

    public function deliveryOrder()
    {
        return $this->belongsTo(DeliveryOrder::class, 'id_delivery_order');
    }

    public function detail()
    {
        return $this->belongsTo(TransactionDetail::class, 'id_detail');
    }

    public function driver()
    {
        return $this->belongsTo(Driver::class, 'id_driver');
    }

    public function truck()
    {
        return $this->belongsTo(Truck::class, 'id_truck');
    }

    /**
     * Generate unique access token for driver
     */
    public static function generateDriverToken()
    {
        do {
            $token = \Illuminate\Support\Str::random(64);
        } while (self::where('driver_access_token', $token)->exists());

        return $token;
    }

    /**
     * Create or update driver access token
     */
    public function assignDriverToken()
    {
        if (!$this->driver_access_token) {
            $this->driver_access_token = self::generateDriverToken();
            $this->save();
        }

        return $this->driver_access_token;
    }

    /**
     * Expire the driver access token
     */
    public function expireToken()
    {
        $this->token_expires_at = now();
        $this->save();
    }

    /**
     * Check if token is valid
     */
    public function isTokenValid()
    {
        if (!$this->driver_access_token) {
            return false;
        }

        if ($this->token_expires_at && $this->token_expires_at->isPast()) {
            return false;
        }

        if (!in_array($this->status, ['pending', 'loading', 'unloading', 'ondelivery', 'onhold'])) {
            return false;
        }

        return true;
    }

    /**
     * Get the driver access URL
     */
    public function getDriverAccessUrl()
    {
        if (!$this->driver_access_token) {
            $this->assignDriverToken();
        }

        return url('/driver/dashboard?token=' . $this->driver_access_token);
    }

    /**
     * Generate WhatsApp message with driver access link
     */
    public function getWhatsAppMessage()
    {
        $url = $this->getDriverAccessUrl();
        $driverName = $this->driver->name ?? 'Driver';
        $doNumber = $this->deliveryOrder->do_number ?? 'N/A';
        $deliveryDate = $this->delivery_date ? $this->delivery_date->format('d/m/Y') : 'N/A';

        $message = "Halo {$driverName},\n\n";
        $message .= "Anda telah ditugaskan untuk pengiriman:\n";
        $message .= "No. DO: {$doNumber}\n";
        $message .= "Tanggal: {$deliveryDate}\n\n";
        $message .= "Silakan akses dashboard Anda melalui link berikut:\n";
        $message .= "{$url}\n\n";
        $message .= "Link ini akan tetap aktif sampai pengiriman selesai.\n";
        $message .= "Terima kasih.";

        return $message;
    }

    /**
     * Get WhatsApp URL with pre-filled message
     * @param string $phoneNumber Phone number in format: 628xxxxxxxxxx
     */
    public function getWhatsAppUrl($phoneNumber = null)
    {
        $phone = $phoneNumber ?? $this->driver->phone ?? '';

        // Clean phone number (remove spaces, dashes, etc)
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // Convert to international format if needed
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }

        $message = urlencode($this->getWhatsAppMessage());

        return "https://wa.me/{$phone}?text={$message}";
    }
}
