<?php

namespace App\Models;

/**
 * Centralized attribute labels for all models.
 * This makes it easier to maintain consistent labels across the application.
 */
class ModelAttributeLabels
{
    /**
     * Get attribute labels for a specific model.
     */
    public static function getLabelsFor(string $modelClass): array
    {
        $modelName = class_basename($modelClass);

        $labels = self::getCommonLabels();

        // Model-specific labels
        $specificLabels = match ($modelName) {
            'Transaction' => self::getTransactionLabels(),
            'TransactionDetail' => self::getTransactionDetailLabels(),
            'Journal' => self::getJournalLabels(),
            'AccountJournal' => self::getAccountJournalLabels(),
            'Invoice' => self::getInvoiceLabels(),
            'InvoiceDetail' => self::getInvoiceDetailLabels(),
            'Price' => self::getPriceLabels(),
            'Route' => self::getRouteLabels(),
            'Customer' => self::getCustomerLabels(),
            'DeliveryOrder' => self::getDeliveryOrderLabels(),
            'DeliverOrderDetail' => self::getDeliverOrderDetailLabels(),
            'Account' => self::getAccountLabels(),
            'Truck' => self::getTruckLabels(),
            'TruckType' => self::getTruckTypeLabels(),
            'TruckLoadout' => self::getTruckLoadoutLabels(),
            'Driver' => self::getDriverLabels(),
            'Location' => self::getLocationLabels(),
            'User' => self::getUserLabels(),
            'Role' => self::getRoleLabels(),
            'Config' => self::getConfigLabels(),
            'AccountCategory' => self::getAccountCategoryLabels(),
            'AccountType' => self::getAccountTypeLabels(),
            'DeliveryOrderFile' => self::getDeliveryOrderFileLabels(),
            'DeliveryOrderStatusLog' => self::getDeliveryOrderStatusLogLabels(),
            default => [],
        };

        return array_merge($labels, $specificLabels);
    }

    /**
     * Common labels used across all models.
     */
    protected static function getCommonLabels(): array
    {
        return [
            'id' => 'ID',
            'code' => 'Code',
            'name' => 'Name',
            'status' => 'Status',
            'notes' => 'Notes',
            'note' => 'Note',
            'description' => 'Description',
            'created_at' => 'Created At',
            'updated_at' => 'Updated At',
            'deleted_at' => 'Deleted At',
            'created_by' => 'Created By',
            'updated_by' => 'Updated By',
            'deleted_by' => 'Deleted By',
        ];
    }

    protected static function getTransactionLabels(): array
    {
        return [
            'id_customer' => 'Customer',
            'id_sales' => 'Sales Person',
            'id_transaction' => 'Parent Transaction',
            'invoice_date' => 'Invoice Date',
            'ppn' => 'PPN (VAT)',
            'ppn_percentage' => 'PPN Percentage',
            'pph' => 'PPH (Tax)',
            'pph_percentage' => 'PPH Percentage',
            'discount' => 'Discount',
            'total' => 'Total',
            'grand_total' => 'Grand Total',
            'type' => 'Type',
        ];
    }

    protected static function getTransactionDetailLabels(): array
    {
        return [
            'id_transaction' => 'Transaction',
            'id_origin' => 'Origin',
            'id_destination' => 'Destination',
            'id_loadout' => 'Truck Loadout',
            'price' => 'Price',
            'pocket_money_1' => 'Pocket Money 1',
            'pocket_money_2' => 'Pocket Money 2',
            'pocket_money_3' => 'Pocket Money 3',
            'bonus' => 'Bonus',
            'sub_total' => 'Sub Total',
            'quantity' => 'Quantity',
            'origin_action' => 'Origin Action',
            'destination_action' => 'Destination Action',
        ];
    }

    protected static function getJournalLabels(): array
    {
        return [
            'date' => 'Date',
            'reference_type' => 'Reference Type',
            'reference_id' => 'Reference ID',
        ];
    }

    protected static function getAccountJournalLabels(): array
    {
        return [
            'id_journal' => 'Journal',
            'id_accounts' => 'Account',
            'credit' => 'Credit',
            'debit' => 'Debit',
        ];
    }

    protected static function getInvoiceLabels(): array
    {
        return [
            'invoice_date' => 'Invoice Date',
            'total' => 'Total',
            'total_discount' => 'Total Discount',
            'total_ppn' => 'Total PPN (VAT)',
            'total_pph' => 'Total PPH (Tax)',
            'grand_total' => 'Grand Total',
            'id_payment_account' => 'Payment Account',
        ];
    }

    protected static function getInvoiceDetailLabels(): array
    {
        return [
            'id_invoice' => 'Invoice',
            'id_transaction' => 'Transaction',
        ];
    }

    protected static function getPriceLabels(): array
    {
        return [
            'id_route' => 'Route',
            'id_loadout' => 'Truck Loadout',
            'id_customer' => 'Customer',
            'price' => 'Price',
        ];
    }

    protected static function getRouteLabels(): array
    {
        return [
            'id_origin' => 'Origin',
            'id_destination' => 'Destination',
            'distance' => 'Distance',
            'duration' => 'Duration',
            'toll' => 'Toll',
            'pocket_money_1' => 'Pocket Money 1',
            'pocket_money_2' => 'Pocket Money 2',
            'pocket_money_3' => 'Pocket Money 3',
            'bonus' => 'Bonus',
        ];
    }

    protected static function getCustomerLabels(): array
    {
        return [
            'full_name' => 'Full Name',
            'phone' => 'Phone',
            'address' => 'Address',
            'email' => 'Email',
            'pic_name' => 'PIC Name',
            'pic_phone' => 'PIC Phone',
            'is_ppn' => 'Subject to PPN',
            'is_pph' => 'Subject to PPH',
            'npwp_no' => 'NPWP Number',
            'term_of_payment' => 'Term of Payment',
            'credit_limit' => 'Credit Limit',
        ];
    }

    protected static function getDeliveryOrderLabels(): array
    {
        return [
            'id_transaction' => 'Transaction',
        ];
    }

    protected static function getDeliverOrderDetailLabels(): array
    {
        return [
            'id_delivery_order' => 'Delivery Order',
            'id_detail' => 'Detail',
            'id_driver' => 'Driver',
            'id_truck' => 'Truck',
        ];
    }

    protected static function getAccountLabels(): array
    {
        return [
            'id_category' => 'Category',
            'id_type' => 'Type',
        ];
    }

    protected static function getTruckLabels(): array
    {
        return [
            'id_type' => 'Truck Type',
            'brand' => 'Brand',
            'body_no' => 'Body Number',
            'machine_no' => 'Machine Number',
            'vehicle_year' => 'Vehicle Year',
            'stnk_owner' => 'STNK Owner',
            'kir_name' => 'KIR Name',
            'license_plate' => 'License Plate',
            'tax_date' => 'Tax Date',
            'kir_date' => 'KIR Date',
            'stnk_date' => 'STNK Date',
            'kir_file' => 'KIR File',
            'bpkb_file' => 'BPKB File',
            'stnk_file' => 'STNK File',
            'credit_start_date' => 'Credit Start Date',
            'credit_end_date' => 'Credit End Date',
            'cctv_gps_no' => 'CCTV/GPS Number',
            'gps_contact' => 'GPS Contact',
        ];
    }

    protected static function getTruckTypeLabels(): array
    {
        return []; // Only common labels
    }

    protected static function getTruckLoadoutLabels(): array
    {
        return []; // Only common labels
    }

    protected static function getDriverLabels(): array
    {
        return [
            'full_name' => 'Full Name',
            'nick_name' => 'Nick Name',
            'phone' => 'Phone',
            'address' => 'Address',
            'identity_number' => 'Identity Number',
            'license_number' => 'License Number',
            'kk_number' => 'KK Number',
            'bpjs_number' => 'BPJS Number',
            'identity_file' => 'Identity File',
            'license_file' => 'License File',
            'license_type' => 'License Type',
            'license_date' => 'License Date',
            'kk_file' => 'KK File',
            'photo_file' => 'Photo File',
            'skck_file' => 'SKCK File',
            'bank_name' => 'Bank Name',
            'bank_owner_name' => 'Bank Owner Name',
            'bank_no' => 'Bank Number',
            'joined_date' => 'Joined Date',
        ];
    }

    protected static function getLocationLabels(): array
    {
        return []; // Only common labels
    }

    protected static function getUserLabels(): array
    {
        return [
            'name' => 'Name',
            'email' => 'Email',
            'password' => 'Password',
            'email_verified_at' => 'Email Verified At',
        ];
    }

    protected static function getRoleLabels(): array
    {
        return [
            'guard_name' => 'Guard Name',
        ];
    }

    protected static function getConfigLabels(): array
    {
        return [
            'key' => 'Key',
            'value' => 'Value',
        ];
    }

    protected static function getAccountCategoryLabels(): array
    {
        return []; // Only common labels
    }

    protected static function getAccountTypeLabels(): array
    {
        return []; // Only common labels
    }

    protected static function getDeliveryOrderFileLabels(): array
    {
        return [
            'id_delivery_order' => 'Delivery Order',
            'file_path' => 'File Path',
            'file_type' => 'File Type',
        ];
    }

    protected static function getDeliveryOrderStatusLogLabels(): array
    {
        return [
            'id_delivery_order' => 'Delivery Order',
            'old_status' => 'Old Status',
            'new_status' => 'New Status',
            'changed_at' => 'Changed At',
            'changed_by' => 'Changed By',
        ];
    }
}
