<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\LogsActivityWithDetails;

class Transaction extends Model
{
    use SoftDeletes, LogsActivityWithDetails;

    protected $fillable = [
        'id_customer',
        'id_sales',
        'id_transaction',
        'code',
        'invoice_date',
        'ppn',
        'ppn_percentage',
        'pph',
        'pph_percentage',
        'discount',
        'total',
        'grand_total',
        'type',
        'status',
        'notes',
        'approved_by',
        'approved_at',
        'rejected_by',
        'rejected_at',
        'created_by',
        'updated_by',
        'deleted_by',
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'approved_at' => 'datetime',
        'rejected_at' => 'datetime',
        'created_at' => 'datetime',
    ];

    /**
     * Get attributes to log - exclude calculated fields.
     */
    protected function getLogAttributes(): array
    {
        // Exclude calculated fields:
        // - ppn (calculated from ppn_percentage)
        // - pph (calculated from pph_percentage)
        // - total (calculated from details)
        // - grand_total (calculated with taxes/discount)
        return array_diff($this->fillable, ['ppn', 'pph', 'total', 'grand_total']);
    }

    /**
     * Get the display name for activity logging based on transaction type.
     */
    protected function getLogName(): string
    {
        return match($this->type) {
            'quotation' => 'Quotation',
            'salesorder' => 'Sales Order',
            default => 'Transaction',
        };
    }

    private $statusColors = [
        'pending' => 'secondary',
        'approved' => 'success',
        'rejected' => 'danger',
        'onprogress' => 'warning',
        'cancel' => 'danger'
    ];

    public function getStatusColor()
    {
        return $this->statusColors[$this->status];
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'id_customer');
    }

    public function sales()
    {
        return $this->belongsTo(User::class, 'id_sales');
    }

    public function parentTransaction()
    {
        return $this->belongsTo(Transaction::class, 'id_transaction');
    }

    public function details()
    {
        return $this->hasMany(TransactionDetail::class, 'id_transaction');
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function rejectedBy()
    {
        return $this->belongsTo(User::class, 'rejected_by');
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function deletedBy()
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }
}
