<?php

namespace App\Observers;

use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\Facades\CauserResolver;

class ModelWithDetailsObserver
{
    protected array $originalDetails = [];
    protected array $detailRelations = [
        'Journal' => 'accountJournals',
        'Invoice' => 'invoiceDetails',
        'Transaction' => 'details',
        'DeliveryOrder' => 'deliverOrderDetails',
    ];

    /**
     * Handle the model "retrieved" event to store original details.
     */
    public function retrieved(Model $model): void
    {
        $this->captureOriginalDetails($model);
    }

    /**
     * Handle the model "updating" event to capture original details.
     */
    public function updating(Model $model): void
    {
        // Capture original details before update
        $this->captureOriginalDetailsForUpdate($model);
    }

    /**
     * Capture original details specifically for update operations.
     */
    protected function captureOriginalDetailsForUpdate(Model $model): void
    {
        $modelClass = class_basename($model);

        if (!isset($this->detailRelations[$modelClass])) {
            return;
        }

        $relationName = $this->detailRelations[$modelClass];

        if (!method_exists($model, $relationName)) {
            return;
        }

        $modelKey = get_class($model) . ':' . ($model->id ?? 'new');

        // Only capture if this model actually exists in DB (has an ID)
        if (!$model->id) {
            $this->originalDetails[$modelKey] = [];
            return;
        }

        // Force reload from database to ensure we get the actual current state
        // Use a fresh query to avoid cached results
        $originalModel = get_class($model)::with($relationName)->find($model->id);

        if ($originalModel) {
            $this->originalDetails[$modelKey] = $originalModel->{$relationName}
                ->map(function ($detail) {
                    return $detail->attributesToArray();
                })
                ->keyBy('id')
                ->toArray();
        } else {
            $this->originalDetails[$modelKey] = [];
        }
    }

    /**
     * Handle the model "updated" event.
     */
    public function updated(Model $model): void
    {
        // For Transaction and Journal models, detail changes are logged manually in the repository
        // after sync completes, so we skip automatic logging here
        // to avoid duplicate or incorrect logs
        $modelClass = class_basename($model);

        if (in_array($modelClass, ['Transaction', 'Journal'])) {
            // Clean up stored original details without logging
            $modelKey = get_class($model) . ':' . ($model->id ?? 'new');
            unset($this->originalDetails[$modelKey]);
            return;
        }

        // For other models, log detail changes normally
        $this->logDetailsChanges($model, 'updated');
    }

    /**
     * Handle the model "created" event.
     */
    public function created(Model $model): void
    {
        // For Transaction and Journal models, detail changes are logged manually in the repository
        // after details are created, so we skip automatic logging here
        $modelClass = class_basename($model);

        if (in_array($modelClass, ['Transaction', 'Journal'])) {
            return;
        }

        // For other models, log detail changes normally
        $this->logDetailsChanges($model, 'created');
    }

    /**
     * Handle the model "deleting" event to capture details before deletion.
     */
    public function deleting(Model $model): void
    {
        $this->captureOriginalDetails($model);
    }

    /**
     * Handle the model "deleted" event.
     */
    public function deleted(Model $model): void
    {
        $this->logDetailsChanges($model, 'deleted');
    }

    /**
     * Capture original details before changes.
     */
    protected function captureOriginalDetails(Model $model): void
    {
        $modelClass = class_basename($model);

        if (!isset($this->detailRelations[$modelClass])) {
            return;
        }

        $relationName = $this->detailRelations[$modelClass];

        if (!method_exists($model, $relationName)) {
            return;
        }

        $modelKey = get_class($model) . ':' . ($model->id ?? 'new');

        // Load the details with all attributes
        $this->originalDetails[$modelKey] = $model->{$relationName}()
            ->get()
            ->map(function ($detail) {
                return $detail->attributesToArray();
            })
            ->keyBy('id')
            ->toArray();
    }

    /**
     * Log changes to details (added, updated, deleted).
     */
    protected function logDetailsChanges(Model $model, string $event): void
    {
        $modelClass = class_basename($model);

        if (!isset($this->detailRelations[$modelClass])) {
            return;
        }

        $relationName = $this->detailRelations[$modelClass];

        if (!method_exists($model, $relationName)) {
            return;
        }

        $modelKey = get_class($model) . ':' . ($model->id ?? 'new');
        $originalDetails = $this->originalDetails[$modelKey] ?? [];

        // For created events, original details should be empty (no previous state)
        // For updated events, we should have captured the original state in the updating event

        // Refresh the model to get current details
        $model->refresh();
        $model->load($relationName);
        $currentDetails = $model->{$relationName}
            ->map(function ($detail) {
                return $detail->attributesToArray();
            })
            ->keyBy('id')
            ->toArray();

        $changes = $this->calculateDetailChanges($originalDetails, $currentDetails);

        // Only log if there are actual detail changes
        if (!empty($changes)) {
            $this->logActivity($model, $event, $changes);
        }

        // Clean up stored original details
        unset($this->originalDetails[$modelKey]);
    }

    /**
     * Calculate changes in details.
     */
    protected function calculateDetailChanges(array $original, array $current): array
    {
        $changes = [
            'added' => [],
            'updated' => [],
            'deleted' => [],
        ];

        // Fields to ignore when comparing (timestamps that change automatically)
        $ignoreFields = ['created_at', 'updated_at', 'deleted_at'];

        // Find added and updated details
        foreach ($current as $id => $currentDetail) {
            if (!isset($original[$id])) {
                // New detail added
                $changes['added'][] = $currentDetail;
            } else {
                // Check for updates, ignoring timestamp fields
                $oldFiltered = array_diff_key($original[$id], array_flip($ignoreFields));
                $newFiltered = array_diff_key($currentDetail, array_flip($ignoreFields));

                $diff = array_diff_assoc($newFiltered, $oldFiltered);

                if (!empty($diff)) {
                    $changes['updated'][] = [
                        'id' => $id,
                        'old' => $original[$id],
                        'new' => $currentDetail,
                        'changes' => $diff,
                    ];
                }
            }
        }

        // Find deleted details
        foreach ($original as $id => $originalDetail) {
            if (!isset($current[$id])) {
                $changes['deleted'][] = $originalDetail;
            }
        }

        // Remove empty change types
        return array_filter($changes, fn($value) => !empty($value));
    }

    /**
     * Log activity with detail changes.
     */
    protected function logActivity(Model $model, string $event, array $detailChanges): void
    {
        $modelName = class_basename($model);
        $identifier = $this->getModelIdentifier($model);

        $description = $this->buildDescription($modelName, $identifier, $event, $detailChanges);

        activity()
            ->performedOn($model)
            ->causedBy(CauserResolver::resolve())
            ->withProperties([
                'detail_changes' => $detailChanges,
                'model_identifier' => $identifier,
            ])
            ->event($event)
            ->log($description);
    }

    /**
     * Build description for activity log.
     */
    protected function buildDescription(string $modelName, string $identifier, string $event, array $detailChanges): string
    {
        $parts = [];

        $parts[] = "{$modelName} {$identifier}";

        if (!empty($detailChanges['added'])) {
            $count = count($detailChanges['added']);
            $parts[] = "{$count} detail(s) added";
        }

        if (!empty($detailChanges['updated'])) {
            $count = count($detailChanges['updated']);
            $parts[] = "{$count} detail(s) updated";
        }

        if (!empty($detailChanges['deleted'])) {
            $count = count($detailChanges['deleted']);
            $parts[] = "{$count} detail(s) deleted";
        }

        return implode(' - ', $parts);
    }

    /**
     * Get model identifier for logging.
     */
    protected function getModelIdentifier(Model $model): string
    {
        if (isset($model->code)) {
            return "'{$model->code}'";
        }

        if (isset($model->name)) {
            return "'{$model->name}'";
        }

        return "#{$model->id}";
    }
}
