<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Vite;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Blade;
use App\Guards\DriverGuard;
use App\Models\Journal;
use App\Models\Invoice;
use App\Models\Transaction;
use App\Models\DeliveryOrder;
use App\Observers\ModelWithDetailsObserver;
use App\Helpers\Helpers;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        // Register helper singleton for facade
        $this->app->singleton('helper', function ($app) {
            return new \App\Helpers\Helpers();
        });

        // Register ModelWithDetailsObserver as singleton
        $this->app->singleton(ModelWithDetailsObserver::class);
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        Vite::useStyleTagAttributes(function (?string $src, string $url, ?array $chunk, ?array $manifest) {
            if ($src !== null) {
                return [
                    'class' => preg_match("/(resources\/assets\/vendor\/scss\/(rtl\/)?core)-?.*/i", $src) ? 'template-customizer-core-css' : (preg_match("/(resources\/assets\/vendor\/scss\/(rtl\/)?theme)-?.*/i", $src) ? 'template-customizer-theme-css' : '')
                ];
            }
            return [];
        });

        // Register custom driver guard
        Auth::extend('driver', function ($app, $name, array $config) {
            return new DriverGuard($app['request']);
        });

        // Register observers for models with details using singleton instance
        $observer = $this->app->make(ModelWithDetailsObserver::class);
        Journal::observe($observer);
        Invoice::observe($observer);
        Transaction::observe($observer);
        DeliveryOrder::observe($observer);

        // Register Blade directive for attribute labels
        Blade::directive('label', function ($expression) {
            return "<?php echo \App\Helpers\Helpers::getAttributeLabel($expression); ?>";
        });

        // Register Blade directive for model labels (with instance)
        Blade::directive('modelLabel', function ($expression) {
            return "<?php echo \App\Helpers\Helpers::modelLabel($expression); ?>";
        });
    }
}
