<?php

namespace App\Repositories;

use App\Repositories\Contracts\RepositoryInterface;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;

abstract class BaseRepository implements RepositoryInterface
{
    /**
     * @var Model
     */
    protected $model;

    /**
     * BaseRepository constructor.
     *
     * @param Model $model
     */
    public function __construct(Model $model)
    {
        $this->model = $model;
    }

    /**
     * Get all records
     *
     * @param array $columns
     * @return Collection
     */
    public function all(array $columns = ['*']): Collection
    {
        return $this->model->all($columns);
    }

    /**
     * Find a record by ID
     *
     * @param int $id
     * @param array $columns
     * @return Model|null
     */
    public function find(int $id, array $columns = ['*']): ?Model
    {
        return $this->model->find($id, $columns);
    }

    /**
     * Find a record by ID or fail
     *
     * @param int $id
     * @param bool $withTrashed
     * @param array $columns
     * @return Model
     */
    public function findOrFail(int $id, bool $withTrashed = false, array $columns = ['*']): Model
    {
        $query = $withTrashed ? $this->model->withTrashed() : $this->model;
        return $query->findOrFail($id, $columns);
    }

    /**
     * Create a new record
     *
     * @param array $data
     * @return Model
     */
    public function create(array $data): Model
    {
        // Automatically add created_by if not already set
        if (!isset($data['created_by']) && in_array('created_by', $this->model->getFillable())) {
            $data['created_by'] = Auth::check() ? Auth::id() : null;
        }

        return $this->model->create($data);
    }

    /**
     * Update a record
     *
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function update(int $id, array $data): bool
    {
        $record = $this->findOrFail($id);

        // Automatically add updated_by if not already set
        if (!isset($data['updated_by']) && in_array('updated_by', $this->model->getFillable())) {
            $data['updated_by'] = Auth::check() ? Auth::id() : null;
        }

        return $record->update($data);
    }

    /**
     * Delete a record
     *
     * @param int $id
     * @return bool
     */
    public function delete(int $id): bool
    {
        $record = $this->findOrFail($id);
        return $record->delete();
    }

    /**
     * Soft delete a record with deleted_by tracking
     *
     * @param int $id
     * @param int|null $deletedBy
     * @return bool
     */
    public function softDelete(int $id, ?int $deletedBy = null): bool
    {
        $record = $this->findOrFail($id);

        // Set deleted_by and status without triggering activity log
        if (in_array('deleted_by', $this->model->getFillable())) {
            // Disable activity logging temporarily
            $record->disableLogging();

            $record->update([
                'deleted_by' => $deletedBy ?? (Auth::check() ? Auth::id() : null),
                'status' => 'cancel'
            ]);

            // Re-enable logging
            $record->enableLogging();

            // Log the cancel action manually
            activity()
                ->performedOn($record)
                ->causedBy(Auth::check() ? Auth::user() : null)
                ->withProperties([
                    'attributes' => [
                        'status' => 'cancel',
                        'deleted_by' => $deletedBy ?? (Auth::check() ? Auth::id() : null),
                    ]
                ])
                ->event('canceled')
                ->log($record->getDescriptionForEvent('canceled'));
        }

        // Disable logging for the delete operation too
        $record->disableLogging();
        return $record->delete();
    }

    /**
     * Get records with pagination
     *
     * @param int $perPage
     * @param array $columns
     * @return mixed
     */
    public function paginate(int $perPage = 15, array $columns = ['*'])
    {
        return $this->model->paginate($perPage, $columns);
    }

    /**
     * Find records by field
     *
     * @param string $field
     * @param mixed $value
     * @param array $columns
     * @return Collection
     */
    public function findBy(string $field, $value, array $columns = ['*']): Collection
    {
        return $this->model->where($field, $value)->get($columns);
    }

    /**
     * Find first record by field
     *
     * @param string $field
     * @param mixed $value
     * @param array $columns
     * @return Model|null
     */
    public function findFirstBy(string $field, $value, array $columns = ['*']): ?Model
    {
        return $this->model->where($field, $value)->first($columns);
    }

    /**
     * Get the model instance
     *
     * @return Model
     */
    public function getModel(): Model
    {
        return $this->model;
    }

    /**
     * Create a new query builder instance
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return $this->model->query();
    }
}
