<?php

namespace App\Repositories;

use App\Models\DeliveryOrder;
use App\Models\DeliverOrderDetail;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DeliveryOrderRepository extends BaseRepository
{
    /**
     * DeliveryOrderRepository constructor.
     *
     * @param DeliveryOrder $model
     */
    public function __construct(DeliveryOrder $model)
    {
        parent::__construct($model);
    }

    /**
     * Create delivery order from sales order
     *
     * @param array $data
     * @param \App\Models\Transaction $salesOrder
     * @return DeliveryOrder
     * @throws \Exception
     */
    public function createDeliveryOrder(array $data, $salesOrder): DeliveryOrder
    {
        DB::beginTransaction();

        try {
            // Generate delivery order code
            $code = $this->generateDeliveryOrderCode();

            // Prepare delivery order data
            $deliveryOrderData = [
                'code' => $code,
                'id_transaction' => $data['id_transaction'],
                'created_by' => Auth::id(),
            ];

            // Create delivery order
            $deliveryOrder = $this->create($deliveryOrderData);

            // Create delivery order details
            if (isset($data['details']) && is_array($data['details'])) {
                foreach ($data['details'] as $detail) {
                    DeliverOrderDetail::create([
                        'id_delivery_order' => $deliveryOrder->id,
                        'id_detail' => $detail['id_detail'],
                        'id_driver' => $detail['id_driver'],
                        'id_truck' => $detail['id_truck'],
                        'qty' => $detail['qty'],
                        'delivery_date' => $detail['delivery_date'],
                        'status' => 'pending',
                        'created_by' => Auth::id(),
                    ]);
                }
            }

            // Update sales order status to onprogress
            $salesOrder->status = 'onprogress';
            $salesOrder->updated_by = Auth::id();
            $salesOrder->save();

            DB::commit();

            return $deliveryOrder->fresh(['transaction', 'details']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Update delivery order with details
     *
     * @param int $id
     * @param array $data
     * @return DeliveryOrder
     * @throws \Exception
     */
    public function updateDeliveryOrder(int $id, array $data): DeliveryOrder
    {
        DB::beginTransaction();

        try {
            $deliveryOrder = $this->findOrFail($id);

            // Prepare delivery order data
            $deliveryOrderData = [
                'id_transaction' => $data['id_transaction'],
                'updated_by' => Auth::id(),
            ];

            // Update delivery order
            $deliveryOrder->update($deliveryOrderData);

            // Sync delivery order details
            $this->syncDeliveryOrderDetails($deliveryOrder->id, $data['details'] ?? []);

            DB::commit();

            return $deliveryOrder->fresh(['transaction', 'details']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Sync delivery order details (update existing, insert new, soft delete removed)
     *
     * @param int $deliveryOrderId
     * @param array $details
     * @return void
     */
    private function syncDeliveryOrderDetails(int $deliveryOrderId, array $details): void
    {
        // Get all existing detail IDs for this delivery order
        $existingIds = DeliverOrderDetail::where('id_delivery_order', $deliveryOrderId)
            ->pluck('id')
            ->toArray();

        // Collect IDs from the incoming details
        $incomingIds = collect($details)
            ->pluck('id')
            ->filter()
            ->toArray();

        // Find IDs that exist in DB but not in incoming data (these should be soft deleted)
        $idsToDelete = array_diff($existingIds, $incomingIds);

        // Soft delete details that are not in the incoming data
        if (!empty($idsToDelete)) {
            DeliverOrderDetail::whereIn('id', $idsToDelete)
                ->update([
                    'deleted_at' => now(),
                    'deleted_by' => Auth::id()
                ]);
        }

        // Process each detail
        foreach ($details as $detail) {
            $detailData = [
                'id_delivery_order' => $deliveryOrderId,
                'id_detail' => $detail['id_detail'],
                'id_driver' => $detail['id_driver'],
                'id_truck' => $detail['id_truck'],
                'qty' => $detail['qty'],
                'delivery_date' => $detail['delivery_date'],
            ];

            if (!empty($detail['id'])) {
                // Update existing detail
                $existingDetail = DeliverOrderDetail::find($detail['id']);
                if ($existingDetail && $existingDetail->id_delivery_order == $deliveryOrderId) {
                    $existingDetail->update(array_merge($detailData, [
                        'updated_by' => Auth::id()
                    ]));
                }
            } else {
                // Insert new detail
                DeliverOrderDetail::create(array_merge($detailData, [
                    'status' => 'pending',
                    'created_by' => Auth::id()
                ]));
            }
        }
    }

    /**
     * Generate unique delivery order code
     *
     * @return string
     */
    private function generateDeliveryOrderCode(): string
    {
        $prefix = 'DO';
        $date = date('Ymd');

        // Get the last delivery order for today
        $lastDeliveryOrder = $this->model
            ->where('code', 'like', $prefix . '/' . $date . '/%')
            ->orderBy('code', 'desc')
            ->first();

        if ($lastDeliveryOrder) {
            // Extract the sequence number and increment
            $parts = explode('/', $lastDeliveryOrder->code);
            $lastSequence = isset($parts[2]) ? intval($parts[2]) : 0;
            $newSequence = $lastSequence + 1;
        } else {
            $newSequence = 1;
        }

        return sprintf('%s/%s/%04d', $prefix, $date, $newSequence);
    }
}
