<?php

namespace App\Repositories;

use App\Models\Journal;
use App\Models\AccountJournal;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class JournalRepository extends BaseRepository
{
    /**
     * JournalRepository constructor.
     *
     * @param Journal $model
     */
    public function __construct(Journal $model)
    {
        parent::__construct($model);
    }

    /**
     * Create journal with account journal details
     *
     * @param array $data
     * @return Journal
     * @throws \Exception
     */
    public function createJournal(array $data): Journal
    {
        DB::beginTransaction();

        try {
            // Generate journal code
            $code = $this->generateJournalCode();

            // Prepare journal data
            $journalData = [
                'code' => $code,
                'date' => $data['date'],
                'reference_type' => 'jurnal_umum',
                'reference_id' => null,
                'description' => $data['description'],
                'created_by' => Auth::id(),
            ];

            // Create journal
            $journal = $this->create($journalData);

            // Create account journal details
            if (isset($data['details']) && is_array($data['details'])) {
                foreach ($data['details'] as $detail) {
                    AccountJournal::create([
                        'id_journal' => $journal->id,
                        'id_accounts' => $detail['id_accounts'],
                        'debit' => $detail['debit'],
                        'credit' => $detail['credit'],
                        'description' => $detail['description'] ?? null,
                        'created_by' => Auth::id(),
                    ]);
                }
            }

            DB::commit();

            return $journal->fresh(['accountJournals.account']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Update journal with account journal details
     *
     * @param int $id
     * @param array $data
     * @return Journal
     * @throws \Exception
     */
    public function updateJournal(int $id, array $data): Journal
    {
        DB::beginTransaction();

        try {
            $journal = $this->findOrFail($id);

            // Capture original details BEFORE any updates
            $journal->load('accountJournals');
            $originalDetails = $journal->accountJournals->map(function ($detail) {
                return $detail->attributesToArray();
            })->keyBy('id')->toArray();

            // Prepare journal data
            $journalData = [
                'date' => $data['date'],
                'description' => $data['description'],
                'updated_by' => Auth::id(),
            ];

            // Update journal (this will fire updating/updated events)
            $journal->update($journalData);

            // Sync account journal details
            $this->syncAccountJournalDetails($journal->id, $data['details'] ?? []);

            // Manually log detail changes after sync
            $this->logDetailChanges($journal, $originalDetails);

            DB::commit();

            return $journal->fresh(['accountJournals.account']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Delete journal with account journal details
     *
     * @param int $id
     * @return bool
     * @throws \Exception
     */
    public function deleteJournal(int $id): bool
    {
        DB::beginTransaction();

        try {
            $journal = $this->findOrFail($id);

            // Delete account journal details
            $journal->accountJournals()->delete();

            // Delete journal
            $result = $journal->delete();

            DB::commit();

            return $result;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Log detail changes to activity log
     *
     * @param Journal $journal
     * @param array $originalDetails
     * @return void
     */
    private function logDetailChanges(Journal $journal, array $originalDetails): void
    {
        // Refresh to get current details after sync
        $journal->refresh();
        $journal->load('accountJournals');

        $currentDetails = $journal->accountJournals->map(function ($detail) {
            return $detail->attributesToArray();
        })->keyBy('id')->toArray();

        // Calculate changes
        $changes = [
            'added' => [],
            'updated' => [],
            'deleted' => [],
        ];

        $ignoreFields = ['created_at', 'updated_at', 'deleted_at'];

        // Find added and updated details
        foreach ($currentDetails as $id => $currentDetail) {
            if (!isset($originalDetails[$id])) {
                // New detail added
                $changes['added'][] = $currentDetail;
            } else {
                // Check for updates, ignoring timestamp fields
                $oldFiltered = array_diff_key($originalDetails[$id], array_flip($ignoreFields));
                $newFiltered = array_diff_key($currentDetail, array_flip($ignoreFields));

                $diff = array_diff_assoc($newFiltered, $oldFiltered);

                if (!empty($diff)) {
                    $changes['updated'][] = [
                        'id' => $id,
                        'old' => $originalDetails[$id],
                        'new' => $currentDetail,
                        'changes' => $diff,
                    ];
                }
            }
        }

        // Find deleted details
        foreach ($originalDetails as $id => $originalDetail) {
            if (!isset($currentDetails[$id])) {
                $changes['deleted'][] = $originalDetail;
            }
        }

        // Remove empty change types
        $changes = array_filter($changes, fn($value) => !empty($value));

        // Only log if there are actual changes
        if (!empty($changes)) {
            $identifier = $journal->code ?? "#{$journal->id}";
            $description = $this->buildDetailChangeDescription($identifier, $changes);

            activity()
                ->performedOn($journal)
                ->causedBy(Auth::id())
                ->withProperties([
                    'detail_changes' => $changes,
                    'model_identifier' => $identifier,
                ])
                ->event('updated')
                ->log($description);
        }
    }

    /**
     * Build description for detail changes
     *
     * @param string $identifier
     * @param array $changes
     * @return string
     */
    private function buildDetailChangeDescription(string $identifier, array $changes): string
    {
        $parts = ["Journal {$identifier}"];

        if (!empty($changes['added'])) {
            $count = count($changes['added']);
            $parts[] = "{$count} detail(s) added";
        }

        if (!empty($changes['updated'])) {
            $count = count($changes['updated']);
            $parts[] = "{$count} detail(s) updated";
        }

        if (!empty($changes['deleted'])) {
            $count = count($changes['deleted']);
            $parts[] = "{$count} detail(s) deleted";
        }

        return implode(' - ', $parts);
    }

    /**
     * Sync account journal details (update existing, insert new, delete removed)
     *
     * @param int $journalId
     * @param array $details
     * @return void
     */
    private function syncAccountJournalDetails(int $journalId, array $details): void
    {
        // Get all existing detail IDs for this journal
        $existingIds = AccountJournal::where('id_journal', $journalId)
            ->pluck('id')
            ->toArray();

        // Collect IDs from the incoming details
        $incomingIds = collect($details)
            ->pluck('id')
            ->filter()
            ->toArray();

        // Find IDs that exist in DB but not in incoming data (these should be deleted)
        $idsToDelete = array_diff($existingIds, $incomingIds);

        // Delete details that are not in the incoming data
        if (!empty($idsToDelete)) {
            AccountJournal::whereIn('id', $idsToDelete)->delete();
        }

        // Process each detail
        foreach ($details as $detail) {
            $detailData = [
                'id_journal' => $journalId,
                'id_accounts' => $detail['id_accounts'],
                'debit' => $detail['debit'],
                'credit' => $detail['credit'],
                'description' => $detail['description'] ?? null,
            ];

            if (!empty($detail['id'])) {
                // Update existing detail only if data has changed
                $existingDetail = AccountJournal::find($detail['id']);
                if ($existingDetail && $existingDetail->id_journal == $journalId) {
                    // Check if any data has actually changed
                    $hasChanges = false;
                    foreach ($detailData as $key => $value) {
                        // Use loose comparison for numeric fields to handle string/number type differences
                        $existingValue = $existingDetail->$key;

                        // Convert both to same type for comparison
                        if (is_numeric($value) && is_numeric($existingValue)) {
                            // Compare as numbers
                            if ((float)$existingValue != (float)$value) {
                                $hasChanges = true;
                                break;
                            }
                        } else {
                            // Compare as strings (handles null properly)
                            if ((string)$existingValue !== (string)$value) {
                                $hasChanges = true;
                                break;
                            }
                        }
                    }

                    // Only update if there are actual changes
                    if ($hasChanges) {
                        $existingDetail->update(array_merge($detailData, [
                            'updated_by' => Auth::id()
                        ]));
                    }
                }
            } else {
                // Insert new detail
                AccountJournal::create(array_merge($detailData, [
                    'created_by' => Auth::id()
                ]));
            }
        }
    }

    /**
     * Generate unique journal code
     *
     * @return string
     */
    private function generateJournalCode(): string
    {
        $lastJournal = Journal::whereYear('created_at', date('Y'))
            ->whereMonth('created_at', date('m'))
            ->orderBy('id', 'desc')
            ->first();

        $monthYear = date('my');
        $sequence = $lastJournal ? (int)substr($lastJournal->code, -4) + 1 : 1;

        return 'JU' . $monthYear . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }
}
