<?php

namespace App\Repositories;

use App\Models\Config;
use App\Models\Customer;
use App\Models\Location;
use App\Models\Price;
use App\Models\Role;
use App\Models\Route;
use App\Models\Transaction;
use App\Models\TransactionDetail;
use App\Models\TruckLoadout;
use App\Models\User;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class QuotationRepository extends BaseRepository
{
    /**
     * QuotationRepository constructor.
     *
     * @param Transaction $model
     */
    public function __construct(Transaction $model)
    {
        parent::__construct($model);
    }

    /**
     * Create quotation with transaction details
     *
     * @param array $data
     * @return Transaction
     * @throws \Exception
     */
    public function createQuotation(array $data): Transaction
    {
        DB::beginTransaction();

        try {
            // Generate quotation code
            $code = $this->generateQuotationCode();

            // If user is SALES role, set id_sales to current user
            $currentUser = Auth::user();
            $currentUserRole = $currentUser->role ? $currentUser->role->name : null;

            if ($currentUserRole === 'SALES') {
                $data['id_sales'] = $currentUser->id;
            }

            // Prepare transaction data
            $transactionData = [
                'code' => $code,
                'id_customer' => $data['id_customer'],
                'id_sales' => $data['id_sales'] ?? null,
                'invoice_date' => $data['invoice_date'],
                'notes' => $data['notes'] ?? null,
                'discount' => $data['discount'] ?? 0,
                'ppn' => $data['ppn'] ?? 0,
                'ppn_percentage' => $data['ppn_percentage'] ?? 0,
                'pph' => $data['pph'] ?? 0,
                'pph_percentage' => $data['pph_percentage'] ?? 0,
                'total' => $data['total'] ?? 0,
                'grand_total' => $data['grand_total'] ?? 0,
                'type' => 'quotation',
                'status' => 'pending',
                'created_by' => Auth::id(),
            ];

            // Create transaction
            $transaction = $this->create($transactionData);

            // Create transaction details
            if (isset($data['details']) && is_array($data['details'])) {
                foreach ($data['details'] as $detail) {
                    // Always get origin and destination names directly from database
                    $originName = null;
                    $destinationName = null;

                    if (isset($detail['id_origin'])) {
                        $origin = \App\Models\Location::find($detail['id_origin']);
                        $originName = $origin ? $origin->name : null;
                    }

                    if (isset($detail['id_destination'])) {
                        $destination = \App\Models\Location::find($detail['id_destination']);
                        $destinationName = $destination ? $destination->name : null;
                    }

                    TransactionDetail::create([
                        'id_transaction' => $transaction->id,
                        'id_origin' => $detail['id_origin'],
                        'origin_name' => $originName,
                        'id_destination' => $detail['id_destination'],
                        'destination_name' => $destinationName,
                        'id_loadout' => $detail['id_loadout'],
                        'price' => $detail['price'],
                        'pocket_money_1' => $detail['pocket_money_1'] ?? 0,
                        'pocket_money_2' => $detail['pocket_money_2'] ?? 0,
                        'pocket_money_3' => $detail['pocket_money_3'] ?? 0,
                        'bonus' => $detail['bonus'] ?? 0,
                        'quantity' => $detail['quantity'],
                        'sub_total' => $detail['sub_total'],
                        'origin_action' => $detail['origin_action'],
                        'destination_action' => $detail['destination_action'],
                        'notes' => $detail['notes'] ?? null,
                        'created_by' => Auth::id(),
                    ]);
                }
            }

            DB::commit();

            return $transaction->fresh(['customer', 'sales']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Update quotation with transaction details
     *
     * @param int $id
     * @param array $data
     * @return Transaction
     * @throws \Exception
     */
    public function updateQuotation(int $id, array $data): Transaction
    {
        DB::beginTransaction();

        try {
            $transaction = $this->findOrFail($id);

            // Capture original details BEFORE any updates
            $transaction->load('details');
            $originalDetails = $transaction->details->map(function ($detail) {
                return $detail->attributesToArray();
            })->keyBy('id')->toArray();

            // Prepare transaction data
            $transactionData = [
                'id_customer' => $data['id_customer'],
                'id_sales' => $data['id_sales'] ?? $transaction->id_sales,
                'invoice_date' => $data['invoice_date'],
                'notes' => $data['notes'] ?? null,
                'discount' => $data['discount'] ?? 0,
                'ppn' => $data['ppn'] ?? 0,
                'ppn_percentage' => $data['ppn_percentage'] ?? 0,
                'pph' => $data['pph'] ?? 0,
                'pph_percentage' => $data['pph_percentage'] ?? 0,
                'total' => $data['total'] ?? 0,
                'grand_total' => $data['grand_total'] ?? 0,
                'updated_by' => Auth::id(),
            ];

            // Update transaction (this will fire updating/updated events)
            $transaction->update($transactionData);

            // Sync transaction details
            $this->syncTransactionDetails($transaction->id, $data['details'] ?? []);

            // Manually log detail changes after sync
            $this->logDetailChanges($transaction, $originalDetails);

            DB::commit();

            return $transaction->fresh(['customer', 'sales']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Log detail changes to activity log
     *
     * @param Transaction $transaction
     * @param array $originalDetails
     * @return void
     */
    private function logDetailChanges(Transaction $transaction, array $originalDetails): void
    {
        // Refresh to get current details after sync
        $transaction->refresh();
        $transaction->load('details');

        $currentDetails = $transaction->details->map(function ($detail) {
            return $detail->attributesToArray();
        })->keyBy('id')->toArray();

        // Calculate changes
        $changes = [
            'added' => [],
            'updated' => [],
            'deleted' => [],
        ];

        $ignoreFields = ['created_at', 'updated_at', 'deleted_at'];

        // Find added and updated details
        foreach ($currentDetails as $id => $currentDetail) {
            if (!isset($originalDetails[$id])) {
                // New detail added
                $changes['added'][] = $currentDetail;
            } else {
                // Check for updates, ignoring timestamp fields
                $oldFiltered = array_diff_key($originalDetails[$id], array_flip($ignoreFields));
                $newFiltered = array_diff_key($currentDetail, array_flip($ignoreFields));

                $diff = array_diff_assoc($newFiltered, $oldFiltered);

                if (!empty($diff)) {
                    $changes['updated'][] = [
                        'id' => $id,
                        'old' => $originalDetails[$id],
                        'new' => $currentDetail,
                        'changes' => $diff,
                    ];
                }
            }
        }

        // Find deleted details
        foreach ($originalDetails as $id => $originalDetail) {
            if (!isset($currentDetails[$id])) {
                $changes['deleted'][] = $originalDetail;
            }
        }

        // Remove empty change types
        $changes = array_filter($changes, fn($value) => !empty($value));

        // Only log if there are actual changes
        if (!empty($changes)) {
            $identifier = $transaction->code ?? "#{$transaction->id}";
            $description = $this->buildDetailChangeDescription($identifier, $changes);

            activity()
                ->performedOn($transaction)
                ->causedBy(Auth::id())
                ->withProperties([
                    'detail_changes' => $changes,
                    'model_identifier' => $identifier,
                ])
                ->event('updated')
                ->log($description);
        }
    }

    /**
     * Build description for detail changes
     *
     * @param string $identifier
     * @param array $changes
     * @return string
     */
    private function buildDetailChangeDescription(string $identifier, array $changes): string
    {
        $parts = ["Quotation {$identifier}"];

        if (!empty($changes['added'])) {
            $count = count($changes['added']);
            $parts[] = "{$count} detail(s) added";
        }

        if (!empty($changes['updated'])) {
            $count = count($changes['updated']);
            $parts[] = "{$count} detail(s) updated";
        }

        if (!empty($changes['deleted'])) {
            $count = count($changes['deleted']);
            $parts[] = "{$count} detail(s) deleted";
        }

        return implode(' - ', $parts);
    }

    /**
     * Sync transaction details (update existing, insert new, soft delete removed)
     *
     * @param int $transactionId
     * @param array $details
     * @return void
     */
    private function syncTransactionDetails(int $transactionId, array $details): void
    {
        // Get all existing detail IDs for this transaction
        $existingIds = TransactionDetail::where('id_transaction', $transactionId)
            ->pluck('id')
            ->toArray();

        // Collect IDs from the incoming details
        $incomingIds = collect($details)
            ->pluck('id')
            ->filter()
            ->toArray();

        // Find IDs that exist in DB but not in incoming data (these should be soft deleted)
        $idsToDelete = array_diff($existingIds, $incomingIds);

        // Soft delete details that are not in the incoming data
        if (!empty($idsToDelete)) {
            TransactionDetail::whereIn('id', $idsToDelete)
                ->update([
                    'deleted_at' => now(),
                    'deleted_by' => Auth::id()
                ]);
        }

        // Process each detail
        foreach ($details as $detail) {
            // Always get origin and destination names directly from database
            $originName = null;
            $destinationName = null;

            if (isset($detail['id_origin'])) {
                $origin = \App\Models\Location::find($detail['id_origin']);
                $originName = $origin ? $origin->name : null;
            }

            if (isset($detail['id_destination'])) {
                $destination = \App\Models\Location::find($detail['id_destination']);
                $destinationName = $destination ? $destination->name : null;
            }

            $detailData = [
                'id_transaction' => $transactionId,
                'id_origin' => $detail['id_origin'],
                'origin_name' => $originName,
                'id_destination' => $detail['id_destination'],
                'destination_name' => $destinationName,
                'id_loadout' => $detail['id_loadout'],
                'price' => $detail['price'],
                'pocket_money_1' => $detail['pocket_money_1'] ?? 0,
                'pocket_money_2' => $detail['pocket_money_2'] ?? 0,
                'pocket_money_3' => $detail['pocket_money_3'] ?? 0,
                'bonus' => $detail['bonus'] ?? 0,
                'quantity' => $detail['quantity'],
                'sub_total' => $detail['sub_total'],
                'origin_action' => $detail['origin_action'],
                'destination_action' => $detail['destination_action'],
                'notes' => $detail['notes'] ?? null,
            ];

            if (!empty($detail['id'])) {
                // Update existing detail
                $existingDetail = TransactionDetail::find($detail['id']);
                if ($existingDetail && $existingDetail->id_transaction == $transactionId) {
                    $existingDetail->update(array_merge($detailData, [
                        'updated_by' => Auth::id()
                    ]));
                }
            } else {
                // Insert new detail
                TransactionDetail::create(array_merge($detailData, [
                    'created_by' => Auth::id()
                ]));
            }
        }
    }

    /**
     * Get tax configuration (PPH and PPN percentages)
     *
     * @return array
     */
    public function getTaxConfig(): array
    {
        $pphConfig = Config::where('key', 'PPH')->first();
        $ppnConfig = Config::where('key', 'PPN')->first();

        return [
            'pph_percentage' => $pphConfig ? (float) $pphConfig->value : 2,
            'ppn_percentage' => $ppnConfig ? (float) $ppnConfig->value : 11,
        ];
    }

    /**
     * Get form data for create/edit views
     *
     * @return array
     */
    public function getFormData(): array
    {
        $customers = Customer::orderBy('full_name')->get();
        $routes = Route::with(['origin', 'destination'])->get();
        $locations = Location::orderBy('name')->get();
        $loadouts = TruckLoadout::orderBy('name')->get();
        $taxConfig = $this->getTaxConfig();

        // Check current user's role
        $currentUser = Auth::user();
        $currentUserRole = $currentUser->role ? $currentUser->role->name : null;
        $isNotSales = $currentUserRole !== 'SALES';

        // Get users with SALES role if current user is not SALES
        $salesUsers = collect();
        if ($isNotSales) {
            $salesRole = Role::where('name', 'SALES')->first();
            if ($salesRole) {
                $salesUsers = User::role('SALES')->orderBy('name')->get();
            }
        }

        return [
            'customers' => $customers,
            'routes' => $routes,
            'locations' => $locations,
            'loadouts' => $loadouts,
            'pphPercentage' => $taxConfig['pph_percentage'],
            'ppnPercentage' => $taxConfig['ppn_percentage'],
            'isNotSales' => $isNotSales,
            'salesUsers' => $salesUsers,
        ];
    }

    /**
     * Calculate quotation totals including taxes
     *
     * @param array $data
     * @return array
     */
    public function calculateQuotationTotals(array $data): array
    {
        // Get customer to check tax settings
        $customer = Customer::findOrFail($data['id_customer']);

        // Get tax configuration
        $taxConfig = $this->getTaxConfig();
        $pphPercentage = $taxConfig['pph_percentage'];
        $ppnPercentage = $taxConfig['ppn_percentage'];

        // Calculate sub_total for each detail (price * quantity)
        $total = 0;
        foreach ($data['details'] as $key => $detail) {
            $subtotal = $detail['price'] * $detail['quantity'];
            $data['details'][$key]['sub_total'] = $subtotal;
            $total += $subtotal;
        }

        // Get discount
        $discount = $data['discount'] ?? 0;
        $afterDiscount = $total - $discount;

        // Calculate PPN and PPH based on customer settings
        $ppn = 0;
        $pph = 0;
        if ($customer->is_ppn) {
            $ppn = $afterDiscount * ($ppnPercentage / 100);
        }
        if ($customer->is_pph) {
            $pph = $afterDiscount * ($pphPercentage / 100);
        }

        // Calculate grand total
        $grandTotal = $afterDiscount + $ppn - $pph;

        return [
            'details' => $data['details'],
            'total' => $total,
            'ppn' => $ppn,
            'ppn_percentage' => $ppnPercentage,
            'pph' => $pph,
            'pph_percentage' => $pphPercentage,
            'grand_total' => $grandTotal,
        ];
    }

    /**
     * Prepare quotation data with calculations
     *
     * @param array $data
     * @return array
     */
    public function prepareQuotationData(array $data): array
    {
        $calculations = $this->calculateQuotationTotals($data);

        return array_merge($data, $calculations);
    }

    /**
     * Get prices for a specific route and customer
     *
     * @param int $routeId
     * @param int|null $customerId
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getPricesForRoute(int $routeId, ?int $customerId = null)
    {
        return Price::with(['route', 'loadout', 'customer'])
            ->where('id_route', $routeId)
            ->where(function ($query) use ($customerId) {
                $query->where('id_customer', $customerId)
                    ->orWhereNull('id_customer');
            })
            ->get();
    }

    /**
     * Get route allowances (pocket money and bonus)
     *
     * @param int $routeId
     * @return array|null
     */
    public function getRouteAllowances(int $routeId): ?array
    {
        $route = Route::find($routeId);

        if (!$route) {
            return null;
        }

        return [
            'pocket_money_1' => $route->pocket_money_1 ?? 0,
            'pocket_money_2' => $route->pocket_money_2 ?? 0,
            'pocket_money_3' => $route->pocket_money_3 ?? 0,
            'bonus' => $route->bonus ?? 0,
        ];
    }

    /**
     * Reject quotation
     *
     * @param int $id
     * @return Transaction
     * @throws \Exception
     */
    public function rejectQuotation(int $id): Transaction
    {
        DB::beginTransaction();

        try {
            $quotation = $this->findOrFail($id);

            // Update quotation status to rejected with custom activity log
            // Disable automatic logging temporarily
            $quotation->disableLogging();
            $quotation->status = 'rejected';
            $quotation->rejected_by = Auth::id();
            $quotation->rejected_at = now();
            $quotation->updated_by = Auth::id();
            $quotation->save();
            $quotation->enableLogging();

            // Log custom activity for rejection
            $identifier = $quotation->code ?? "#{$quotation->id}";
            activity()
                ->performedOn($quotation)
                ->causedBy(Auth::id())
                ->event('rejected')
                ->log("Reject Quotation {$identifier}");

            DB::commit();

            return $quotation->fresh();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Generate unique quotation code
     *
     * @return string
     */
    private function generateQuotationCode(): string
    {
        $prefix = 'QUO';
        $date = date('Ymd');

        // Get the last quotation for today
        $lastQuotation = $this->model
            ->where('code', 'like', $prefix . '/' . $date . '/%')
            ->orderBy('code', 'desc')
            ->first();

        if ($lastQuotation) {
            // Extract the sequence number and increment
            $parts = explode('/', $lastQuotation->code);
            $lastSequence = isset($parts[2]) ? intval($parts[2]) : 0;
            $newSequence = $lastSequence + 1;
        } else {
            $newSequence = 1;
        }

        return sprintf('%s/%s/%04d', $prefix, $date, $newSequence);
    }
}
