<?php

namespace App\Repositories;

use App\Models\Role;
use App\Repositories\BaseRepository;
use Spatie\Permission\Models\Permission;

class RoleRepository extends BaseRepository
{
    /**
     * RoleRepository constructor.
     *
     * @param Role $model
     */
    public function __construct(Role $model)
    {
        parent::__construct($model);
    }

    /**
     * Get role with permissions
     *
     * @param int $id
     * @return Role
     */
    public function findWithPermissions(int $id): Role
    {
        return $this->model->with('permissions')->findOrFail($id);
    }

    /**
     * Sync permissions with role
     *
     * @param Role $role
     * @param array $permissionNames
     * @return void
     */
    public function syncPermissions(Role $role, array $permissionNames): void
    {
        // Capture original permissions before sync
        $originalPermissions = $role->permissions->pluck('name')->toArray();

        $permissions = [];
        foreach ($permissionNames as $permissionName) {
            $permission = Permission::firstOrCreate(
                ['name' => $permissionName, 'guard_name' => 'web']
            );
            $permissions[] = $permission->id;
        }

        $role->permissions()->sync($permissions);

        // Log permission changes
        $this->logPermissionChanges($role, $originalPermissions, $permissionNames);
    }

    /**
     * Log permission changes to activity log
     *
     * @param Role $role
     * @param array $originalPermissions
     * @param array $newPermissions
     * @return void
     */
    private function logPermissionChanges(Role $role, array $originalPermissions, array $newPermissions): void
    {
        // Calculate changes
        $addedPermissions = array_diff($newPermissions, $originalPermissions);
        $removedPermissions = array_diff($originalPermissions, $newPermissions);

        // Only log if there are changes
        if (empty($addedPermissions) && empty($removedPermissions)) {
            return;
        }

        // Group related permissions (create+store, edit+update) to count as one
        $addedGrouped = $this->groupRelatedPermissions($addedPermissions);
        $removedGrouped = $this->groupRelatedPermissions($removedPermissions);

        $changes = [];
        if (!empty($addedPermissions)) {
            $changes['added'] = array_values($addedPermissions);
            $changes['added_count'] = $addedGrouped; // Store actual count for display
        }
        if (!empty($removedPermissions)) {
            $changes['removed'] = array_values($removedPermissions);
            $changes['removed_count'] = $removedGrouped; // Store actual count for display
        }

        $description = $this->buildPermissionChangeDescription($role->name, $changes);

        activity()
            ->performedOn($role)
            ->causedBy(auth()->id())
            ->withProperties([
                'permission_changes' => $changes,
                'role_name' => $role->name,
            ])
            ->event('updated')
            ->log($description);
    }

    /**
     * Group related permissions to count them as one
     * For example: create+store = 1, edit+update = 1
     *
     * @param array $permissions
     * @return int
     */
    private function groupRelatedPermissions(array $permissions): int
    {
        $groups = [];
        $relatedActions = [
            'create' => 'store',
            'store' => 'create',
            'edit' => 'update',
            'update' => 'edit',
        ];

        foreach ($permissions as $permission) {
            // Split permission into resource and action (e.g., "customer.create" -> ["customer", "create"])
            $parts = explode('.', $permission);
            if (count($parts) !== 2) {
                $groups[$permission] = true;
                continue;
            }

            [$resource, $action] = $parts;

            // Check if this action has a related action
            if (isset($relatedActions[$action])) {
                $relatedAction = $relatedActions[$action];
                $relatedPermission = "{$resource}.{$relatedAction}";

                // If the related permission is also in the list, group them together
                if (in_array($relatedPermission, $permissions)) {
                    // Use a consistent key for both (alphabetically first)
                    $groupKey = $action < $relatedAction
                        ? "{$resource}.{$action}"
                        : "{$resource}.{$relatedAction}";
                    $groups[$groupKey] = true;
                } else {
                    // No related permission, count individually
                    $groups[$permission] = true;
                }
            } else {
                // No related action, count individually
                $groups[$permission] = true;
            }
        }

        return count($groups);
    }

    /**
     * Build description for permission changes
     *
     * @param string $roleName
     * @param array $changes
     * @return string
     */
    private function buildPermissionChangeDescription(string $roleName, array $changes): string
    {
        $parts = ["Role '{$roleName}' permissions updated"];

        if (!empty($changes['added'])) {
            // Use grouped count if available, otherwise count all permissions
            $count = $changes['added_count'] ?? count($changes['added']);
            $parts[] = "{$count} permission(s) added";
        }

        if (!empty($changes['removed'])) {
            // Use grouped count if available, otherwise count all permissions
            $count = $changes['removed_count'] ?? count($changes['removed']);
            $parts[] = "{$count} permission(s) removed";
        }

        return implode(' - ', $parts);
    }

    /**
     * Get all roles with permission count
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function queryWithPermissionCount()
    {
        return $this->model->withCount('permissions');
    }
}
