<?php

namespace App\Repositories;

use App\Models\Transaction;
use App\Models\TransactionDetail;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SalesOrderRepository extends BaseRepository
{
    /**
     * SalesOrderRepository constructor.
     *
     * @param Transaction $model
     */
    public function __construct(Transaction $model)
    {
        parent::__construct($model);
    }

    /**
     * Create sales order from quotation
     *
     * @param array $data
     * @param Transaction $quotation
     * @return Transaction
     * @throws \Exception
     */
    public function createSalesOrder(array $data, Transaction $quotation): Transaction
    {
        DB::beginTransaction();

        try {
            // Generate sales order code
            $code = $this->generateSalesOrderCode();

            // Prepare transaction data
            $transactionData = [
                'code' => $code,
                'id_customer' => $data['id_customer'],
                'id_sales' => $data['id_sales'] ?? null,
                'id_transaction' => $quotation->id, // Link to quotation
                'invoice_date' => $data['invoice_date'],
                'notes' => $data['notes'] ?? null,
                'discount' => $data['discount'] ?? 0,
                'ppn' => $data['ppn'] ?? 0,
                'ppn_percentage' => $data['ppn_percentage'] ?? 0,
                'pph' => $data['pph'] ?? 0,
                'pph_percentage' => $data['pph_percentage'] ?? 0,
                'total' => $data['total'] ?? 0,
                'grand_total' => $data['grand_total'] ?? 0,
                'type' => 'salesorder',
                'status' => 'pending',
                'created_by' => Auth::id(),
            ];

            // Create transaction
            $transaction = $this->create($transactionData);

            // Create transaction details
            if (isset($data['details']) && is_array($data['details'])) {
                foreach ($data['details'] as $detail) {
                    // Always get origin and destination names directly from database
                    $originName = null;
                    $destinationName = null;

                    if (isset($detail['id_origin'])) {
                        $origin = \App\Models\Location::find($detail['id_origin']);
                        $originName = $origin ? $origin->name : null;
                    }

                    if (isset($detail['id_destination'])) {
                        $destination = \App\Models\Location::find($detail['id_destination']);
                        $destinationName = $destination ? $destination->name : null;
                    }

                    TransactionDetail::create([
                        'id_transaction' => $transaction->id,
                        'id_origin' => $detail['id_origin'],
                        'origin_name' => $originName,
                        'id_destination' => $detail['id_destination'],
                        'destination_name' => $destinationName,
                        'id_loadout' => $detail['id_loadout'],
                        'price' => $detail['price'],
                        'pocket_money_1' => $detail['pocket_money_1'] ?? 0,
                        'pocket_money_2' => $detail['pocket_money_2'] ?? 0,
                        'pocket_money_3' => $detail['pocket_money_3'] ?? 0,
                        'bonus' => $detail['bonus'] ?? 0,
                        'quantity' => $detail['quantity'],
                        'sub_total' => $detail['sub_total'],
                        'origin_action' => $detail['origin_action'],
                        'destination_action' => $detail['destination_action'],
                        'notes' => $detail['notes'] ?? null,
                        'created_by' => Auth::id(),
                    ]);
                }
            }

            // Update quotation status to approved with custom activity log
            // Disable automatic logging temporarily
            $quotation->disableLogging();
            $quotation->status = 'approved';
            $quotation->approved_by = Auth::id();
            $quotation->approved_at = now();
            $quotation->updated_by = Auth::id();
            $quotation->save();
            $quotation->enableLogging();

            // Log custom activity for approval
            $identifier = $quotation->code ?? "#{$quotation->id}";
            activity()
                ->performedOn($quotation)
                ->causedBy(Auth::id())
                ->event('approved')
                ->log("Approve Quotation {$identifier}");

            DB::commit();

            return $transaction->fresh(['customer', 'sales', 'parentTransaction']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Update sales order with transaction details
     *
     * @param int $id
     * @param array $data
     * @return Transaction
     * @throws \Exception
     */
    public function updateSalesOrder(int $id, array $data): Transaction
    {
        DB::beginTransaction();

        try {
            $transaction = $this->findOrFail($id);

            // Prepare transaction data
            $transactionData = [
                'id_customer' => $data['id_customer'],
                'id_sales' => $data['id_sales'] ?? $transaction->id_sales,
                'invoice_date' => $data['invoice_date'],
                'notes' => $data['notes'] ?? null,
                'discount' => $data['discount'] ?? 0,
                'ppn' => $data['ppn'] ?? 0,
                'ppn_percentage' => $data['ppn_percentage'] ?? 0,
                'pph' => $data['pph'] ?? 0,
                'pph_percentage' => $data['pph_percentage'] ?? 0,
                'total' => $data['total'] ?? 0,
                'grand_total' => $data['grand_total'] ?? 0,
                'updated_by' => Auth::id(),
            ];

            // Update transaction
            $transaction->update($transactionData);

            // Sync transaction details
            $this->syncTransactionDetails($transaction->id, $data['details'] ?? []);

            DB::commit();

            return $transaction->fresh(['customer', 'sales', 'parentTransaction']);
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Sync transaction details (update existing, insert new, soft delete removed)
     *
     * @param int $transactionId
     * @param array $details
     * @return void
     */
    private function syncTransactionDetails(int $transactionId, array $details): void
    {
        // Get all existing detail IDs for this transaction
        $existingIds = TransactionDetail::where('id_transaction', $transactionId)
            ->pluck('id')
            ->toArray();

        // Collect IDs from the incoming details
        $incomingIds = collect($details)
            ->pluck('id')
            ->filter()
            ->toArray();

        // Find IDs that exist in DB but not in incoming data (these should be soft deleted)
        $idsToDelete = array_diff($existingIds, $incomingIds);

        // Soft delete details that are not in the incoming data
        if (!empty($idsToDelete)) {
            TransactionDetail::whereIn('id', $idsToDelete)
                ->update([
                    'deleted_at' => now(),
                    'deleted_by' => Auth::id()
                ]);
        }

        // Process each detail
        foreach ($details as $detail) {
            // Always get origin and destination names directly from database
            $originName = null;
            $destinationName = null;

            if (isset($detail['id_origin'])) {
                $origin = \App\Models\Location::find($detail['id_origin']);
                $originName = $origin ? $origin->name : null;
            }

            if (isset($detail['id_destination'])) {
                $destination = \App\Models\Location::find($detail['id_destination']);
                $destinationName = $destination ? $destination->name : null;
            }

            $detailData = [
                'id_transaction' => $transactionId,
                'id_origin' => $detail['id_origin'],
                'origin_name' => $originName,
                'id_destination' => $detail['id_destination'],
                'destination_name' => $destinationName,
                'id_loadout' => $detail['id_loadout'],
                'price' => $detail['price'],
                'pocket_money_1' => $detail['pocket_money_1'] ?? 0,
                'pocket_money_2' => $detail['pocket_money_2'] ?? 0,
                'pocket_money_3' => $detail['pocket_money_3'] ?? 0,
                'bonus' => $detail['bonus'] ?? 0,
                'quantity' => $detail['quantity'],
                'sub_total' => $detail['sub_total'],
                'origin_action' => $detail['origin_action'],
                'destination_action' => $detail['destination_action'],
                'notes' => $detail['notes'] ?? null,
            ];

            if (!empty($detail['id'])) {
                // Update existing detail
                $existingDetail = TransactionDetail::find($detail['id']);
                if ($existingDetail && $existingDetail->id_transaction == $transactionId) {
                    $existingDetail->update(array_merge($detailData, [
                        'updated_by' => Auth::id()
                    ]));
                }
            } else {
                // Insert new detail
                TransactionDetail::create(array_merge($detailData, [
                    'created_by' => Auth::id()
                ]));
            }
        }
    }

    /**
     * Generate unique sales order code
     *
     * @return string
     */
    private function generateSalesOrderCode(): string
    {
        $prefix = 'SO';
        $date = date('Ymd');

        // Get the last sales order for today
        $lastSalesOrder = $this->model
            ->where('code', 'like', $prefix . '/' . $date . '/%')
            ->orderBy('code', 'desc')
            ->first();

        if ($lastSalesOrder) {
            // Extract the sequence number and increment
            $parts = explode('/', $lastSalesOrder->code);
            $lastSequence = isset($parts[2]) ? intval($parts[2]) : 0;
            $newSequence = $lastSequence + 1;
        } else {
            $newSequence = 1;
        }

        return sprintf('%s/%s/%04d', $prefix, $date, $newSequence);
    }
}
