<?php

namespace App\Repositories;

use App\Models\User;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Hash;

class UserRepository extends BaseRepository
{
    /**
     * UserRepository constructor.
     *
     * @param User $model
     */
    public function __construct(User $model)
    {
        parent::__construct($model);
    }

    /**
     * Create user with hashed password and assign role
     *
     * @param array $data
     * @return User
     */
    public function createUser(array $data): User
    {
        // Extract role_id before creating user
        $roleId = $data['role_id'] ?? null;
        unset($data['role_id']);

        if (isset($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }

        $user = $this->create($data);

        // Assign role using Spatie (by ID)
        if ($roleId) {
            $user->roles()->sync([$roleId]);
        }

        return $user;
    }

    /**
     * Update user with optional password hashing and role sync
     *
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateUser(int $id, array $data): bool
    {
        // Extract role_id before updating user
        $roleId = $data['role_id'] ?? null;
        unset($data['role_id']);

        if (isset($data['password']) && !empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']);
        }

        $result = $this->update($id, $data);

        // Sync role using Spatie (by ID)
        if ($roleId) {
            $user = $this->findOrFail($id);
            $user->roles()->sync([$roleId]);
        }

        return $result;
    }

    /**
     * Find user by username
     *
     * @param string $username
     * @return User|null
     */
    public function findByUsername(string $username): ?User
    {
        return $this->findFirstBy('username', $username);
    }
}
