<?php
// app/Traits/DataTableTrait.php

namespace App\Traits;

use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;

trait DataTableTrait
{

  private $icons = [
    "view" => "icon-base ti tabler-eye icon-22px",
    "edit" => "icon-base ti tabler-edit icon-22px",
    "delete" => "icon-base ti tabler-trash icon-22px"
  ];

  /**
   * Get permission name for an action type
   */
  private function getPermissionForAction($resource, $actionType)
  {
    $permissionMap = [
      'view' => 'show',
      'edit' => 'update',
      'delete' => 'destroy'
    ];

    $permission = $permissionMap[$actionType] ?? $actionType;
    return "{$resource}.{$permission}";
  }

  /**
   * Add permissions to actions array
   */
  protected function addPermissionsToActions($actions, $resource)
  {
    return array_map(function($action) use ($resource) {
      if (!isset($action['permission'])) {
        $action['permission'] = $this->getPermissionForAction($resource, $action['type']);
      }
      return $action;
    }, $actions);
  }

  /**
   * Process DataTable request
   */
  protected function processDataTable(Request $request, Builder $query, array $columns, array $actions, array $filters, array $formatters = [])
  {
    $totalRecords = $query->count();

    // Apply custom filters
    if ($request->filters) {
      $query = $this->applyFilters($request, $query, $filters);
    }

    $totalFiltered = $query->count();

    // Detect relationships from columns and get their foreign keys
    $relationNames = [];
    $foreignKeys = [];

    foreach ($columns as $column) {
      $data = $column['data'];

      if (isset($data["relation"])) {
        $relationName = $data["relation"];

        $relationNames[] = $relationName;
      }
    }

    // Apply eager loading for relationships
    if (!empty($relationNames)) {
      $query->with($relationNames);
    }

    // Select only columns specified in $columns array (excluding DT_RowIndex, action, and relationship columns)
    $selectColumns = array_filter(array_map(function ($column) {
      $data = $column['data'];
      return (!in_array($data, ['DT_RowIndex', 'action']) && !isset($column["exclude_column"])) ? $data : null;
    }, $columns));

    // Always include 'id' if not already present (needed for actions)
    if (!in_array('id', $selectColumns)) {
      $selectColumns[] = 'id';
    }

    // Include foreign keys for relationships
    foreach ($foreignKeys as $foreignKey) {
      if (!in_array($foreignKey, $selectColumns)) {
        $selectColumns[] = $foreignKey;
      }
    }

    if (!empty($selectColumns)) {
      $query->select($selectColumns);
    }

    // Apply ordering
    if ($request->has('order')) {
      $orderColumnIndex = $request->input('order.0.column');
      $orderDirection = $request->input('order.0.dir', 'asc');

      if (isset($columns[$orderColumnIndex]['data'])) {
        $orderColumn = $columns[$orderColumnIndex]['data'];
        $query->orderBy($orderColumn, $orderDirection);
      }
    }

    // Apply pagination
    $start = $request->input('start', 0);
    $length = $request->input('length', 10);

    if ($length != -1) {
      $query->skip($start)->take($length);
    }

    $data = $query->get();

    // Map data based on columns
    $data = $data->map(function ($row) use ($columns, $formatters) {
      $result = [];

      $result["id"] = $row["id"];

      // Get model casts for automatic date formatting
      $casts = $row->getCasts();

      foreach ($columns as $column) {
        $key = $column['data'];

        if (isset($column["relation"])) {

          $relationName = $column["relation"];
          $relationColumn = $column["relation_column"];

          $result[$key] = $row->$relationName ? $row->$relationName->$relationColumn : null;
        }
        // Handle regular columns
        else if (!isset($column["relation"])) {
          $value = $row->$key ?? null;

          // Auto-format dates based on model casts
          if ($value && isset($casts[$key])) {
            $castType = $casts[$key];

            // Handle datetime casts
            if ($castType === 'datetime' || $key === 'created_at' || $key === 'updated_at') {
              $result[$key] = date("Y-m-d H:i:s", strtotime($value));
            }
            // Handle date casts
            else if (str_starts_with($castType, 'date')) {
              $result[$key] = date("Y-m-d", strtotime($value));
            }
            else {
              $result[$key] = $value;
            }
          } else {
            $result[$key] = $value;
          }
        }

        // Apply custom formatter if exists for this column
        if (isset($formatters[$key]) && is_callable($formatters[$key])) {
          $result[$key] = $formatters[$key]($row, $result[$key]);
        }
      }

      // Store original row data for condition checking (prefixed to avoid conflicts)
      $result['_originalRow'] = $row;

      return $result;
    });

    $data = $this->processAction($data, $actions);

    return [
      'draw' => intval($request->input('draw')),
      'recordsTotal' => $totalRecords,
      'recordsFiltered' => $totalFiltered,
      'data' => $data
    ];
  }

  protected function processAction($result, $actions)
  {
    return $result = $result->map(function ($row) use ($actions) {

      $action_list = '';

      // Get original row for condition checking
      $originalRow = $row['_originalRow'] ?? null;

      foreach ($actions as $action) {
        // Check if permission key exists and if user has permission
        if (isset($action['permission'])) {
          // Skip this action if user doesn't have permission
          if (!auth()->user()->can($action['permission'])) {
            continue;
          }
        }

        // Check if condition exists and if row meets the condition
        if (isset($action['condition']) && $originalRow) {
          $conditionMet = true;

          foreach ($action['condition'] as $field => $allowedValues) {
            // Get the value from the original row (not formatted)
            $rowValue = $originalRow->$field ?? null;

            // If the field value is not in the allowed values, skip this action
            // Use loose comparison to handle type differences
            if (!in_array($rowValue, $allowedValues, false)) {
              $conditionMet = false;
              break;
            }
          }

          // Skip this action if condition is not met
          if (!$conditionMet) {
            continue;
          }
        }

        $action_list .= '
          <a href="#" data-action="' . $action["type"] . '" data-url="' . route($action["url"], $row["id"]) . '" data-id="' . $row["id"] . '" class="' . $action["type"] . ' btn btn-text-dark rounded-pill waves-effect btn-icon btn-action">
            <i class="' . $this->icons[$action["type"]] . '"></i>
          </a>
        ';
      }

      $row['action'] = '<div class="d-flex align-items-center">' . $action_list . '</div>';

      // Remove the original row data from the response
      unset($row['_originalRow']);

      return $row;
    });
  }

  /**
   * Apply custom filters to query
   */
  protected function applyFilters(Request $request, Builder $query, array $filters): Builder
  {
    // Check if filters exist in request
    if (!$request->has('filters')) {
      return $query;
    }

    $requestFilters = $request->input('filters', []);

    // If filters is empty, return query as is
    if (empty($requestFilters)) {
      return $query;
    }

    foreach ($filters as $filter) {
      $filterName = $filter['name'];
      $filterType = $filter['type'];
      $filterValue = $requestFilters[$filterName] ?? null;

      // Skip if filter value is empty, null, or whitespace only
      if (is_null($filterValue) || $filterValue === '' || trim($filterValue) === '') {
        continue;
      }

      // Get column name (use 'column' if provided, otherwise use filter name)
      $column = $filter['column'] ?? $filterName;

      // Check if this is a relationship filter
      if (isset($filter['relation'])) {
        $relation = $filter['relation'];
        $relationColumn = $filter['relation_column'] ?? 'id';

        // Apply whereHas for relationship filtering
        $query->whereHas($relation, function ($q) use ($relationColumn, $filterValue, $filterType) {
          if ($filterType === 'select') {
            $q->where($relationColumn, '=', $filterValue);
          } else {
            $q->where($relationColumn, 'like', '%' . $filterValue . '%');
          }
        });
      }
      // Apply filter based on type
      else if ($filterType === 'select') {
        // Use exact match for select
        $query->where($column, '=', $filterValue);
      } else {
        // Use LIKE for text, number, date, daterange
        $query->where($column, 'like', '%' . $filterValue . '%');
      }
    }

    return $query;
  }
}
