<?php

namespace App\Traits;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Log;
use Throwable;

trait HandlesExceptions
{
    /**
     * Report exception with enriched context data
     *
     * @param Throwable $e
     * @param string $message
     * @return void
     */
    protected function reportException(Throwable $e, string $message): void
    {
        // Log with full context for Telescope
        Log::error($message, [
            'exception' => get_class($e),
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'url' => request()->fullUrl(),
            'method' => request()->method(),
            'user_id' => auth()->id() ?? 'guest',
            'ip' => request()->ip(),
            'payload' => request()->except(['password', 'password_confirmation', 'token', '_token']),
            'user_agent' => request()->userAgent(),
            'trace' => $e->getTraceAsString(),
        ]);

        // Also report to exception handler
        report($e);
    }

    /**
     * Handle exception for web requests (redirects)
     *
     * @param Throwable $e
     * @param string $message
     * @param string $redirectRoute
     * @return RedirectResponse
     */
    protected function handleException(Throwable $e, string $message = 'An error occurred', string $redirectRoute = null): RedirectResponse
    {
        // Report to Telescope/logs with full context
        $this->reportException($e, $message);

        $redirect = $redirectRoute
            ? redirect()->route($redirectRoute)
            : redirect()->back()->withInput();

        return $redirect->with('error', $message . ': ' . $e->getMessage());
    }

    /**
     * Handle exception for API/AJAX requests (JSON response)
     *
     * @param Throwable $e
     * @param string $message
     * @param int $statusCode
     * @return JsonResponse
     */
    protected function handleJsonException(Throwable $e, string $message = 'An error occurred', int $statusCode = 500): JsonResponse
    {
        // Report to Telescope/logs with full context
        $this->reportException($e, $message);

        return response()->json([
            'success' => false,
            'message' => $message . ': ' . $e->getMessage()
        ], $statusCode);
    }
}
