<?php

namespace App\Traits;

use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

trait LogsActivityWithDetails
{
    use LogsActivity;

    /**
     * Get the options for logging activity.
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly($this->getLogAttributes())
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs()
            ->setDescriptionForEvent(fn(string $eventName) => $this->getDescriptionForEvent($eventName))
            ->useLogName($this->getLogName());
    }

    /**
     * Customize the activity before it's saved.
     * This ensures the causer is always set correctly.
     */
    public function tapActivity(\Spatie\Activitylog\Contracts\Activity $activity, string $eventName): void
    {
        // Ensure causer is set to the authenticated user
        if (auth()->check() && !$activity->causer_id) {
            $activity->causer_id = auth()->id();
            $activity->causer_type = get_class(auth()->user());
        }
    }

    /**
     * Get attributes to log.
     */
    protected function getLogAttributes(): array
    {
        // Default to all fillable attributes
        return $this->fillable;
    }

    /**
     * Get the log name for this model.
     */
    protected function getLogName(): string
    {
        return class_basename($this);
    }

    /**
     * Customize the description for activity log.
     */
    public function getDescriptionForEvent(string $eventName): string
    {
        $modelName = $this->getLogName();
        $identifier = $this->getModelIdentifier();

        return match($eventName) {
            'created' => "{$modelName} {$identifier} has been created",
            'updated' => "{$modelName} {$identifier} has been updated",
            'deleted' => "{$modelName} {$identifier} has been deleted",
            'canceled' => "{$modelName} {$identifier} has been canceled",
            default => "{$modelName} {$identifier} has been {$eventName}",
        };
    }

    /**
     * Get model identifier for logging.
     */
    protected function getModelIdentifier(): string
    {
        // Try to use 'code' or 'name' or fall back to ID
        if (isset($this->code)) {
            return "'{$this->code}'";
        }

        if (isset($this->name)) {
            return "'{$this->name}'";
        }

        return "#{$this->id}";
    }

    /**
     * Get attribute labels for human-readable display.
     * Override this method in your model to customize labels.
     */
    public function getAttributeLabels(): array
    {
        // Try to get labels from centralized ModelAttributeLabels class
        if (class_exists(\App\Models\ModelAttributeLabels::class)) {
            return \App\Models\ModelAttributeLabels::getLabelsFor(get_class($this));
        }

        // Fallback to basic labels
        return [
            'id' => 'ID',
            'created_at' => 'Created At',
            'updated_at' => 'Updated At',
            'deleted_at' => 'Deleted At',
            'created_by' => 'Created By',
            'updated_by' => 'Updated By',
            'deleted_by' => 'Deleted By',
        ];
    }

    /**
     * Get human-readable label for an attribute.
     */
    public function getAttributeLabel(string $attribute): string
    {
        $labels = $this->getAttributeLabels();

        if (isset($labels[$attribute])) {
            return $labels[$attribute];
        }

        // Convert snake_case to Title Case
        return ucwords(str_replace('_', ' ', $attribute));
    }

    /**
     * Get human-readable value for an attribute.
     * This handles foreign keys by fetching related model data.
     */
    public function getReadableAttributeValue(string $attribute, $value)
    {
        // Handle null values
        if ($value === null) {
            return 'N/A';
        }

        // Handle boolean values
        if (is_bool($value)) {
            return $value ? 'Yes' : 'No';
        }

        // Handle dates
        if (in_array($attribute, ['created_at', 'updated_at', 'deleted_at']) && $value instanceof \DateTime) {
            return $value->format('Y-m-d H:i:s');
        }

        // Handle foreign keys - fetch the related model directly using the ID value
        if (preg_match('/^id_(.+)/', $attribute, $matches)) {
            $relationName = \Illuminate\Support\Str::camel($matches[1]);

            if (method_exists($this, $relationName)) {
                try {
                    // Get the relation instance to determine the related model class
                    $relation = $this->$relationName();

                    if ($relation instanceof \Illuminate\Database\Eloquent\Relations\BelongsTo) {
                        $relatedModelClass = get_class($relation->getRelated());

                        // Fetch the related model directly by ID to avoid using cached relationship
                        $related = $relatedModelClass::find($value);

                        if ($related) {
                            // Try common identifier fields
                            if (isset($related->name)) {
                                return $related->name;
                            }
                            if (isset($related->code)) {
                                return $related->code;
                            }
                            if (isset($related->full_name)) {
                                return $related->full_name;
                            }
                            return "ID: {$value}";
                        }
                    }
                } catch (\Exception $e) {
                    return "ID: {$value}";
                }
            }
            return "ID: {$value}";
        }

        return $value;
    }
}
