/**
 * Global Notification Helper
 * Initializes Notyf for toast notifications across the application
 */

'use strict';

// Custom Notyf class to allow HTML content in messages
class CustomNotyf extends Notyf {
  _renderNotification(options) {
    const notification = super._renderNotification(options);

    // Replace textContent with innerHTML to render HTML content
    if (options.message) {
      notification.message.innerHTML = options.message;
    }

    return notification;
  }
}

// Initialize global Notyf instance
window.notyf = new CustomNotyf({
  duration: 4000,
  ripple: true,
  dismissible: true,
  position: { x: 'right', y: 'top' },
  types: [
    {
      type: 'info',
      background: config.colors.info,
      className: 'notyf__info',
      icon: {
        className: 'icon-base ti tabler-info-circle-filled icon-md text-white',
        tagName: 'i'
      }
    },
    {
      type: 'warning',
      background: config.colors.warning,
      className: 'notyf__warning',
      icon: {
        className: 'icon-base ti tabler-alert-triangle-filled icon-md text-white',
        tagName: 'i'
      }
    },
    {
      type: 'success',
      background: config.colors.success,
      className: 'notyf__success',
      icon: {
        className: 'icon-base ti tabler-circle-check-filled icon-md text-white',
        tagName: 'i'
      }
    },
    {
      type: 'error',
      background: config.colors.danger,
      className: 'notyf__error',
      icon: {
        className: 'icon-base ti tabler-xbox-x-filled icon-md text-white',
        tagName: 'i'
      }
    }
  ]
});

/**
 * Show success notification
 * @param {string} message - Message to display
 * @param {number} duration - Duration in milliseconds (optional)
 */
window.showSuccessNotification = function (message, duration = 4000) {
  window.notyf.open({
    type: 'success',
    message: message,
    duration: duration
  });
};

/**
 * Show error notification
 * @param {string} message - Message to display
 * @param {number} duration - Duration in milliseconds (optional)
 */
window.showErrorNotification = function (message, duration = 5000) {
  window.notyf.open({
    type: 'error',
    message: message,
    duration: duration
  });
};

/**
 * Show info notification
 * @param {string} message - Message to display
 * @param {number} duration - Duration in milliseconds (optional)
 */
window.showInfoNotification = function (message, duration = 4000) {
  window.notyf.open({
    type: 'info',
    message: message,
    duration: duration
  });
};

/**
 * Show warning notification
 * @param {string} message - Message to display
 * @param {number} duration - Duration in milliseconds (optional)
 */
window.showWarningNotification = function (message, duration = 4000) {
  window.notyf.open({
    type: 'warning',
    message: message,
    duration: duration
  });
};

/**
 * Handle Laravel session flash messages on page load
 * Automatically shows notifications for success, error, warning, and info messages
 */
document.addEventListener('DOMContentLoaded', function () {
  // Check for success message
  const successMessage = document.querySelector('meta[name="flash-success"]');
  if (successMessage) {
    showSuccessNotification(successMessage.content);
  }

  // Check for error message
  const errorMessage = document.querySelector('meta[name="flash-error"]');
  if (errorMessage) {
    showErrorNotification(errorMessage.content);
  }

  // Check for warning message
  const warningMessage = document.querySelector('meta[name="flash-warning"]');
  if (warningMessage) {
    showWarningNotification(warningMessage.content);
  }

  // Check for info message
  const infoMessage = document.querySelector('meta[name="flash-info"]');
  if (infoMessage) {
    showInfoNotification(infoMessage.content);
  }
});
